// src/app/api/annualtrainingschedule/annualschedule.ts

import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Helper: Get exact error message from FastAPI
const getBackendErrorMessage = (error: any): string => {
  return (
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.response?.data?.error ||
    error?.message ||
    "Something went wrong. Please try again."
  );
};

// AUTH HEADERS
const getAuthHeaders = () => {
  let token = localStorage.getItem("authToken") || "";
  if (token && !token.startsWith("Bearer ")) {
    token = `Bearer ${token}`;
  }

  return {
    headers: {
      Authorization: token,
      Accept: "application/json",
      "Content-Type": "application/x-www-form-urlencoded",
    },
  };
};

// CREATE SCHEDULE
export const createAnnualSchedule = async (year: string) => {
  try {
    const body = new URLSearchParams({ year });
    const res = await axios.post(
      `${API_BASE_URL}/annual-training/create`,
      body,
      getAuthHeaders()
    );
    return res.data;
  } catch (error: any) {
    toast.error(getBackendErrorMessage(error));
    throw error;
  }
};

// GET ALL SCHEDULES
export const getAllAnnualSchedules = async (): Promise<any[]> => {
  try {
    const res = await axios.get(
      `${API_BASE_URL}/annual-training`,
      getAuthHeaders()
    );

    if (Array.isArray(res.data)) return res.data;
    if (res.data?.data && Array.isArray(res.data.data)) return res.data.data;
    if (res.data?.Datas && Array.isArray(res.data.Datas)) return res.data.Datas;
    if (res.data?.result && Array.isArray(res.data.result))
      return res.data.result;

    console.warn("Unexpected response format:", res.data);
    return [];
  } catch (error: any) {
    toast.error(getBackendErrorMessage(error));
    throw error;
  }
};

// DELETE SCHEDULE
export const deleteAnnualSchedule = async (scheduleCode: string) => {
  try {
    const res = await axios.delete(
      `${API_BASE_URL}/annual-training/${scheduleCode}`,
      getAuthHeaders()
    );
    return res.data;
  } catch (error: any) {
    toast.error(getBackendErrorMessage(error));
    throw error;
  }
};

// APPROVE / REJECT
export const approveAnnualSchedule = async (
  scheduleCode: string,
  status: string,
  reviewed_by_id: string,
  rejection_reason?: string
) => {
  try {
    const body = new URLSearchParams({
      status,
      reviewed_by_id,
      rejection_reason: rejection_reason || "",
    });

    const res = await axios.put(
      `${API_BASE_URL}/annual-training/approve/${scheduleCode}`,
      body,
      getAuthHeaders()
    );
    return res.data;
  } catch (error: any) {
    toast.error(getBackendErrorMessage(error));
    throw error;
  }
};

// ADD TRAINING ITEM
export const addTrainingItem = async (
  scheduleCode: string,
  data: {
    training_id: string;
    planned_month: string;
    trainer_id?: string;
    planned_date?: string;
    remarks?: string;
  }
) => {
  try {
    const body = new URLSearchParams({
      training_id: data.training_id,
      planned_month: data.planned_month,
      trainer_id: data.trainer_id || "",
      planned_date: data.planned_date || "",
      remarks: data.remarks || "",
    });

    const res = await axios.post(
      `${API_BASE_URL}/annual-training/${scheduleCode}/add-item`,
      body,
      getAuthHeaders()
    );
    return res.data;
  } catch (error: any) {
    toast.error(getBackendErrorMessage(error));
    throw error;
  }
};

// UPDATE TRAINING ITEM
export const updateTrainingItem = async (
  trainingItemId: string,
  data: {
    training_id?: string;
    planned_month?: string;
    planned_date?: string;
    trainer_id?: string;
    attendance_status?: string;
    remarks?: string;
  }
) => {
  try {
    const body = new URLSearchParams();
    if (data.training_id) body.append("training_id", data.training_id);
    if (data.planned_month) body.append("planned_month", data.planned_month);
    if (data.planned_date) body.append("planned_date", data.planned_date);
    if (data.trainer_id) body.append("trainer_id", data.trainer_id);
    if (data.attendance_status)
      body.append("attendance_status", data.attendance_status);
    if (data.remarks) body.append("remarks", data.remarks);

    const res = await axios.put(
      `${API_BASE_URL}/annual-training/item/${trainingItemId}/update-plan`,
      body,
      getAuthHeaders()
    );
    return res.data;
  } catch (error: any) {
    toast.error(getBackendErrorMessage(error));
    throw error;
  }
};

// DELETE TRAINING ITEM
export const deleteTrainingItem = async (trainingItemId: string) => {
  try {
    const res = await axios.delete(
      `${API_BASE_URL}/annual-training/item/${trainingItemId}`,
      getAuthHeaders()
    );
    return res.data;
  } catch (error: any) {
    toast.error(getBackendErrorMessage(error));
    throw error;
  }
};

// GET SCHEDULE ITEMS
export const getScheduleItems = async (scheduleCode: string) => {
  try {
    const res = await axios.get(
      `${API_BASE_URL}/annual-training/${scheduleCode}/items`,
      getAuthHeaders()
    );
    return res.data;
  } catch (error: any) {
    toast.error(getBackendErrorMessage(error));
    throw error;
  }
};

// GET ALL TRAINERS
export const getAllTrainers = async () => {
  try {
    const res = await axios.get(
      `${API_BASE_URL}/get-manage-user`,
      getAuthHeaders()
    );

    const users = res.data?.Datas || res.data?.data || [];
    return users.map((u: any) => ({
      id: u.id,
      name:
        `${u.First_name || ""} ${u.Last_name || ""}`.trim() || "Unknown User",
    }));
  } catch (error: any) {
    toast.error(getBackendErrorMessage(error));
    throw error;
  }
};
