"use client";

import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// All interfaces (exactly as you wrote – NOT touched)
export interface InductionSchedule {
  trainee_signature: File;
  hr_signature: File;
  trainer_signature: File;
  trainer_id: string;
  date: string;
  duration: string;
  scope: string;
  topics_covered: string;
  hr_signed_date: string;
  employee_code: string;
}

export interface UpdateInductionSchedule {
  trainee_signature?: File;
  trainer_signature?: File;
  from_time?: string;
  to_time?: string;
  trainer_id: string;
  date: string;
  duration: string;
  scope: string;
  topics_covered: string;
  hr_signed_date?: string;
}

export interface UpdateSignaturesData {
  hod_signature?: File;
  hod_remarks?: string;
  hod_signed_date?: string;
  evaluation_understanding_org?: boolean;
  evaluation_understanding_qms?: boolean;
  evaluation_job_requirements?: boolean;
  evaluation_internal_customers?: boolean;
}

export interface InductionScheduleResponse {
  status: string;
  message: string;
  data: {
    schedule_code: string;
    employee_code: string;
  };
}

export interface UpdateInductionScheduleResponse {
  status: string;
  message: string;
}

export interface UpdateSignaturesResponse {
  status: string;
  message: string;
}

export interface DeleteInductionScheduleResponse {
  status: string;
  message: string;
}

export interface InductionScheduleListItem {
  schedule_code: string;
  employee_code: string;
  employee_name: string;
  designation_emp: string;
  department_emp: string;
  scope: string;
  date: string;
  duration: string;
  trainer_id: string;
  trainer_name: string;
  from_time?: string | null;
  to_time?: string | null;
  topics_covered: string;
  trainer_signature?: string | null;
  trainee_signature?: string | null;
  hod_signature?: string | null;
  hr_signature?: string | null;
  evaluation_understanding_org?: boolean | null;
  evaluation_understanding_qms?: boolean | null;
  evaluation_job_requirements?: boolean | null;
  evaluation_internal_customers?: boolean | null;
  hod_remarks?: string | null;
  hod_signed_date?: string | null;
  hr_signed_date?: string | null;
  created_by: string;
  created_by_name: string;
  updated_by?: string | null;
  updated_by_name?: string | null;
  created_at: string;
  updated_at: string;
}

export interface GetAllInductionResponse {
  status: string;
  count: number;
  data: InductionScheduleListItem[];
}

export interface GetInductionByIdResponse {
  status: string;
  data: InductionScheduleListItem;
}

// Get auth headers (your original Bearer fix – kept 100%)
const getAuthHeaders = () => {
  let token = localStorage.getItem("authToken") || "";
  token = token.replace(/^Bearer\s+/i, "");
  return {
    headers: {
      Authorization: `Bearer ${token}`,
      Accept: "application/json",
    },
  };
};

// Helper functions (your original – untouched)
const buildInductionForm = (data: InductionSchedule): FormData => {
  const formData = new FormData();
  formData.append("trainee_signature", data.trainee_signature);
  formData.append("hr_signature", data.hr_signature);
  formData.append("trainer_signature", data.trainer_signature);
  formData.append("trainer_id", data.trainer_id);
  formData.append("date", data.date);
  formData.append("duration", data.duration);
  formData.append("scope", data.scope);
  formData.append("topics_covered", data.topics_covered);
  formData.append("hr_signed_date", data.hr_signed_date);
  formData.append("employee_code", data.employee_code);
  return formData;
};

const buildUpdateForm = (data: UpdateInductionSchedule): FormData => {
  const formData = new FormData();
  if (data.trainee_signature)
    formData.append("trainee_signature", data.trainee_signature);
  if (data.trainer_signature)
    formData.append("trainer_signature", data.trainer_signature);
  if (data.from_time) formData.append("from_time", data.from_time);
  if (data.to_time) formData.append("to_time", data.to_time);
  formData.append("trainer_id", data.trainer_id);
  formData.append("date", data.date);
  formData.append("duration", data.duration);
  formData.append("scope", data.scope);
  formData.append("topics_covered", data.topics_covered);
  if (data.hr_signed_date)
    formData.append("hr_signed_date", data.hr_signed_date);
  return formData;
};

const buildSignaturesForm = (data: UpdateSignaturesData): FormData => {
  const formData = new FormData();
  if (data.hod_signature) formData.append("hod_signature", data.hod_signature);
  if (data.hod_remarks) formData.append("hod_remarks", data.hod_remarks);
  if (data.hod_signed_date)
    formData.append("hod_signed_date", data.hod_signed_date);
  if (data.evaluation_understanding_org !== undefined)
    formData.append(
      "evaluation_understanding_org",
      String(data.evaluation_understanding_org)
    );
  if (data.evaluation_understanding_qms !== undefined)
    formData.append(
      "evaluation_understanding_qms",
      String(data.evaluation_understanding_qms)
    );
  if (data.evaluation_job_requirements !== undefined)
    formData.append(
      "evaluation_job_requirements",
      String(data.evaluation_job_requirements)
    );
  if (data.evaluation_internal_customers !== undefined)
    formData.append(
      "evaluation_internal_customers",
      String(data.evaluation_internal_customers)
    );
  return formData;
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.response?.data?.error ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE INDUCTION SCHEDULE
export const createInductionSchedule = async (
  data: InductionSchedule
): Promise<InductionScheduleResponse> => {
  try {
    const response = await axios.post(
      `${API_BASE_URL}/create_induction_schedule`,
      buildInductionForm(data),
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL INDUCTION SCHEDULES
export const getAllInductionSchedules =
  async (): Promise<GetAllInductionResponse> => {
    try {
      const response = await axios.get(
        `${API_BASE_URL}/get_all_induction_schedule`,
        getAuthHeaders()
      );
      return response.data;
    } catch (error: any) {
      showBackendError(error);
      throw error;
    }
  };

// GET INDUCTION SCHEDULE BY ID
export const getInductionScheduleById = async (
  scheduleCode: string
): Promise<GetInductionByIdResponse> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get_induction_schedule/${scheduleCode}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE INDUCTION SCHEDULE
export const updateInductionSchedule = async (
  scheduleCode: string,
  data: UpdateInductionSchedule
): Promise<UpdateInductionScheduleResponse> => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update_induction_schedule/${scheduleCode}`,
      buildUpdateForm(data),
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE SIGNATURES + EVALUATIONS
export const updateSignatures = async (
  scheduleCode: string,
  data: UpdateSignaturesData
): Promise<UpdateSignaturesResponse> => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update_signatures/${scheduleCode}`,
      buildSignaturesForm(data),
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE INDUCTION SCHEDULE
export const deleteInductionSchedule = async (
  scheduleCode: string
): Promise<DeleteInductionScheduleResponse> => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete_induction_schedule/${scheduleCode}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD INDUCTION SCHEDULE FILE
export const downloadInductionFile = async (
  scheduleCode: string,
  fileType: "trainee_sign" | "trainer_sign" | "hr_sign" | string
): Promise<void> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-schedule-file/${scheduleCode}?file_type=${fileType}`,
      {
        ...getAuthHeaders(),
        responseType: "blob",
      }
    );

    const contentDisposition = response.headers["content-disposition"];
    const filenameMatch = contentDisposition?.match(/filename="(.+)"/);
    const filename = filenameMatch
      ? filenameMatch[1]
      : `${scheduleCode}_${fileType}.xlsx`;

    const blob = new Blob([response.data], {
      type: "application/octet-stream",
    });
    const url = window.URL.createObjectURL(blob);
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", filename);
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    window.URL.revokeObjectURL(url);
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
