"use client";

import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Manage User Interface (exactly as you wrote – NOT touched)
export interface ManageUser {
  id?: string;
  email: string;
  phone_number: string;
  password: string;
  Gender: string;
  DOB: string;
  First_name: string;
  Last_name: string;
  role: string;
  emp_status?: string;
  employee_role?: string;
  department?: string;
  manager?: string;
  join_date?: string;
  location?: string;
  address?: string;
  country?: string;
  qualification?: string;
  experience_years?: string;
  certifications?: string;
  photo?: File | string | null;

  // system fields
  created_at?: string;
  updated_at?: string;
  last_activity?: string;
  is_delete?: boolean;
}

// Auth headers helper (your original – perfect Bearer fix kept)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  const authHeader = token.startsWith("Bearer ") ? token : `Bearer ${token}`;

  return {
    headers: {
      Authorization: authHeader,
      Accept: "application/json",
    },
  };
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.response?.data?.error ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE MANAGE USER
export const createManageUser = async (data: ManageUser) => {
  try {
    const form = new FormData();

    form.append("email", data.email);
    form.append("phone_number", data.phone_number);
    form.append("password", data.password);
    form.append("Gender", data.Gender);
    form.append("DOB", data.DOB);
    form.append("First_name", data.First_name);
    form.append("Last_name", data.Last_name);
    form.append("role", data.role);
    form.append("emp_status", data.emp_status || "Active");

    if (data.manager) form.append("manager", data.manager);
    if (data.department) form.append("department", data.department);
    if (data.join_date) form.append("join_date", data.join_date);
    if (data.location) form.append("location", data.location);
    if (data.address) form.append("address", data.address);
    if (data.country) form.append("country", data.country);
    if (data.qualification) form.append("qualification", data.qualification);
    if (data.experience_years)
      form.append("experience_years", data.experience_years);
    if (data.certifications) form.append("certifications", data.certifications);

    if (data.photo && data.photo instanceof File) {
      form.append("photo", data.photo);
    }

    const response = await axios.post(
      `${API_BASE_URL}/create-manage-user`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL MANAGE USERS
export const getAllManageUsers = async (): Promise<ManageUser[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-manage-user`,
      getAuthHeaders()
    );
    return response.data?.Datas ?? [];
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET MANAGE USER BY ID
export const getManageUserById = async (
  userId: string,
  token?: string
): Promise<ManageUser> => {
  try {
    const headers: any = {
      "Content-Type": "application/json",
      Accept: "application/json",
      Authorization: token
        ? token.startsWith("Bearer ")
          ? token
          : `Bearer ${token}`
        : getAuthHeaders().headers.Authorization,
    };

    const response = await axios.get(
      `${API_BASE_URL}/get-manage-user/${userId}`,
      {
        headers,
      }
    );

    if (!response.data) {
      throw new Error("No data returned from API");
    }

    return response.data?.Datas || response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE MANAGE USER
export const updateManageUser = async (
  userId: string,
  data: Partial<ManageUser>
) => {
  try {
    const form = new FormData();

    Object.entries(data).forEach(([key, value]) => {
      if (value !== undefined && value !== null) {
        if (key === "photo" && value instanceof File) {
          form.append("photo", value);
        } else {
          form.append(key, String(value));
        }
      }
    });

    const response = await axios.put(
      `${API_BASE_URL}/update-manage-user/${userId}`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE MANAGE USER
export const deleteManageUser = async (userId: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-manage-user/${userId}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
