// app/api/performance-appraisal.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

/* ---------------------------------------------
 * Interfaces (ALL FIELDS FROM BACKEND INCLUDED)
 * --------------------------------------------- */

export interface CreateAppraisalData {
  employee_code: string;
  appraisal_period: string;
  hr_reviewer_id: string;
  quality_of_work: number;
  on_time_delivery: number;
  teamwork: number;
  innovation?: number;
  leadership?: number;
  remarks?: string;
  hr_remarks?: string;
  new_training_recommended?: boolean;
}

export interface AppraisalRecord {
  appraisal_code: string;
  employee_id: number;
  employee_code: string;
  employee_name: string;
  appraisal_period: string;
  hr_reviewer_id: string;
  hr_reviewer_name: string;
  approved_by_id?: string | null;
  approved_by_name?: string | null;

  quality_of_work: number;
  on_time_delivery: number;
  teamwork: number;
  innovation?: number;
  leadership?: number;

  training_completed: number;
  effectiveness_score: number;
  competency_score: number;
  overall_rating: number;

  remarks?: string;
  hr_remarks?: string;
  new_training_recommended: boolean;

  created_at: string;
  updated_at: string;
}

export interface CreateAppraisalResponse {
  status: string;
  message: string;
  appraisal_code: string;
  auto_values: {
    training_completed: number;
    effectiveness_score: number;
    competency_score: number;
    overall_rating: number;
  };
}

export interface GetAllAppraisalsResponse {
  status: string;
  count: number;
  data: AppraisalRecord[];
}

export interface GetAppraisalByCodeResponse {
  status: string;
  data: AppraisalRecord;
}

export interface UpdateAppraisalData {
  quality_of_work?: number;
  on_time_delivery?: number;
  teamwork?: number;
  innovation?: number;
  leadership?: number;
  remarks?: string;
  hr_remarks?: string;
  new_training_recommended?: boolean;
}

export interface UpdateAppraisalResponse {
  status: string;
  message: string;
  data: {
    training_completed: number;
    effectiveness_score: number;
    competency_score: number;
    overall_rating: number;
  };
}

export interface ApproveAppraisalData {
  approver_id: string;
}

export interface ApproveAppraisalResponse {
  status: string;
  message: string;
  approved_by: string;
  approved_user_id: string;
  approved_at: string;
}

export interface DeleteAppraisalResponse {
  status: string;
  message: string;
}

/* ---------------------------
 * Helpers
 * --------------------------- */

const getAuthHeaders = () => {
  let token = localStorage.getItem("authToken") || "";

  if (!token.startsWith("Bearer ")) {
    token = `Bearer ${token}`;
  }

  return {
    headers: {
      Authorization: token,
      "Content-Type": "application/x-www-form-urlencoded",
      Accept: "application/json",
    },
  };
};

// Convert any object → URLSearchParams
const toForm = (data: any): URLSearchParams => {
  const form = new URLSearchParams();
  Object.entries(data).forEach(([key, value]: any) => {
    if (value !== undefined && value !== null) {
      form.append(key, String(value));
    }
  });
  return form;
};

const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.response?.data?.error ||
    error?.message ||
    "Something went wrong";

  toast.error(msg);
};

/* ---------------------------
 * API FUNCTIONS
 * --------------------------- */

// CREATE appraisal
export const createAppraisal = async (
  data: CreateAppraisalData
): Promise<CreateAppraisalResponse> => {
  try {
    const response = await axios.post(
      `${API_BASE_URL}/create_appraisal`,
      toForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL appraisals
export const getAllAppraisals = async (): Promise<AppraisalRecord[]> => {
  try {
    const response = await axios.get<GetAllAppraisalsResponse>(
      `${API_BASE_URL}/get_all_appraisals`,
      getAuthHeaders()
    );
    return response.data.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET appraisal by code
export const getAppraisalByCode = async (
  appraisalCode: string
): Promise<AppraisalRecord> => {
  try {
    const response = await axios.get<GetAppraisalByCodeResponse>(
      `${API_BASE_URL}/get_appraisal/${appraisalCode}`,
      getAuthHeaders()
    );
    return response.data.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE appraisal
export const updateAppraisal = async (
  appraisalCode: string,
  data: UpdateAppraisalData
): Promise<UpdateAppraisalResponse> => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update_appraisal/${appraisalCode}`,
      toForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// APPROVE appraisal
export const approveAppraisal = async (
  appraisalCode: string,
  data: ApproveAppraisalData
): Promise<ApproveAppraisalResponse> => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/approve_appraisal/${appraisalCode}`,
      toForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE appraisal
export const deleteAppraisal = async (
  appraisalCode: string
): Promise<DeleteAppraisalResponse> => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete_appraisal/${appraisalCode}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
