// lib/api/supplier-approval.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Interface (exactly as you wrote – NOT touched)
export interface SupplierApproval {
  supplier_id: string;
  supplier_name?: string;
  supplier_type?: string;
  contactPerson?: string;
  email?: string | null;
  phone_number?: string | null;
  product_id: string;
  product_name?: string;
  product_code?: string;
  risk_level?: string;
  inspectionResult?: string;
  rejectionReason?: string;
  testResultDetails?: string | null;
  testedBy?: string;
  testDate?: string;
  status?: string;
  created_at?: string;
  updated_at?: string;
  approved_at?: string | null;
  created_by_id?: string;
  created_by_name?: string;
  updated_by_id?: string;
  updated_by_name?: string;
  approved_by_id?: string | null;
  approved_by_name?: string | null;
}

const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      Accept: "application/json",
    },
  };
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// GET ALL SUPPLIER APPROVALS
export const getAllSupplierApprovals = async (): Promise<
  SupplierApproval[]
> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-all-for-approval-suppliertests`,
      getAuthHeaders()
    );
    return response.data.DATA;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET SUPPLIER APPROVAL BY SUPPLIER ID + PRODUCT ID
export const getSupplierApprovalByIds = async (
  supplierId: string,
  productId: string
): Promise<SupplierApproval> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-suppliertests-for-approval-by-supplier-id-product-id/${supplierId}-${productId}`,
      getAuthHeaders()
    );
    return response.data.DATA;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD SUPPLIER APPROVAL DOCUMENT
export const downloadSupplierApprovalDocument = async (
  supplierId: string,
  productId: string
) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-document-for-approval-suppliertest/${supplierId}-${productId}`,
      {
        ...getAuthHeaders(),
        responseType: "blob",
      }
    );

    // Extract filename if backend sends it
    const contentDisposition = response.headers["content-disposition"];
    let fileName = `SupplierTest_${supplierId}_${productId}.pdf`;
    if (contentDisposition) {
      const match = contentDisposition.match(/filename="(.+?)"/);
      if (match?.[1]) fileName = match[1];
    }

    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", fileName);
    document.body.appendChild(link);
    link.click();
    link.remove();

    return response;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// APPROVE / REJECT SUPPLIER TEST
export const approveSupplierTest = async (
  supplierId: string,
  productId: string,
  status: "Draft" | "Submitted" | "Approved" | "Rejected"
) => {
  try {
    const params = new URLSearchParams();
    params.append("Status", status);

    const response = await axios.put(
      `${API_BASE_URL}/approve-suppliertest/${supplierId}-${productId}`,
      params,
      {
        ...getAuthHeaders(),
        headers: {
          "Content-Type": "application/x-www-form-urlencoded",
          Authorization: localStorage.getItem("authToken") || "",
          Accept: "application/json",
        },
      }
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE SUPPLIER APPROVAL TEST
export const deleteSupplierApprovalTest = async (
  supplierId: string,
  productId: string
) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-suppliertest-OnApproval/${supplierId}-${productId}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
