// app/api/supplier-assessment.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";
import dayjs from "dayjs";
import relativeTime from "dayjs/plugin/relativeTime";

dayjs.extend(relativeTime);

const API_BASE_URL = API_BASE_PATH;

// Full Interface for Supplier Assessment 
export interface SupplierAssessment {
  supplier_id: string;
  companyName?: string;
  supplier_type?: string;
  license_numbers?: string;
  address?: string;
  phone_number?: string;
  country?: string;
  fax?: string;
  email?: string;

  objective?: string;
  total_persons_employed?: number;
  no_of_shifts?: number;
  no_of_shifts_per_person?: number;
  assessment_date: string;
  assessment_type: string;
  auditor_name: string;
  date_of_audit?: string | null;
  key_person_audited?: string;
  position?: string;
  signature?: string | null;

  type_of_products_manufactured?: string;
  product_details?: string;
  supplier_registration_done?: string;
  supplier_register_now_current?: string;
  major_customer_details?: string;
  installed_capacity_details?: string;
  plant_machinery_details?: string;

  documented_procedure?: string;
  documented_procedure_comment?: string;
  follows_gmp?: string;
  follows_gmp_comment?: string;
  has_policy_objectives?: string;
  has_policy_objectives_comment?: string;
  manufacturing_process?: string;
  manufacturing_process_comment?: string;
  system_for_spec_control?: string;
  system_for_spec_control_comment?: string;
  has_our_specs_test_methods?: string;
  has_our_specs_test_methods_comment?: string;
  customer_spec_control?: string;
  customer_spec_control_comment?: string;
  customer_property_handling?: string;
  customer_property_handling_comment?: string;
  raw_materials?: string;
  raw_materials_comment?: string;
  raw_material_suppliers?: string;
  raw_material_suppliers_comment?: string;
  has_supplier_list?: string;
  has_supplier_list_comment?: string;
  raw_material_rejected?: string;
  raw_material_rejected_comment?: string;
  corrective_action_for_rejection?: string;
  corrective_action_for_rejection_comment?: string;
  quality_checks?: string;
  quality_checks_comment?: string;
  personnel_competent?: string;
  personnel_competent_comment?: string;
  adequate_testing_resources?: string;
  adequate_testing_resources_comment?: string;
  measuring_equipment_details?: string;
  measuring_equipment_details_comment?: string;
  calibrates_equipment?: string;
  calibrates_equipment_comment?: string;
  equipment_status_displayed?: string;
  equipment_status_displayed_comment?: string;
  equipment_identified?: string;
  equipment_identified_comment?: string;
  calibration_records_maintained?: string;
  calibration_records_maintained_comment?: string;
  batch_numbering_method?: string;
  batch_numbering_method_comment?: string;
  batch_coding_system?: string;
  batch_coding_system_comment?: string;
  has_traceability?: string;
  has_traceability_comment?: string;
  gives_certificate_of_analysis?: string;
  gives_certificate_of_analysis_comment?: string;
  finished_product_label_control?: string;
  finished_product_label_control_comment?: string;
  process_control_adequate?: string;
  process_control_adequate_comment?: string;
  documented_inspection_procedure?: string;
  documented_inspection_procedure_comment?: string;
  inspection_records_maintained?: string;
  inspection_records_maintained_comment?: string;
  process_control_details?: string;
  process_control_details_comment?: string;
  identifies_product_status?: string;
  identifies_product_status_comment?: string;
  adequate_to_avoid_mixup?: string;
  adequate_to_avoid_mixup_comment?: string;
  special_conditions_affect_quality?: string;
  special_conditions_affect_quality_comment?: string;
  contamination_control_method?: string;
  contamination_control_method_comment?: string;
  records_maintained_each_stage?: string;
  records_maintained_each_stage_comment?: string;
  record_retention_period?: string;
  record_retention_period_comment?: string;
  records_traceable?: string;
  records_traceable_comment?: string;
  maintain_test_records?: string;
  maintain_test_records_comment?: string;
  inspection_test_records_up_to_date?: string;
  inspection_test_records_up_to_date_comment?: string;
  testing_for_finished_goods?: string;
  testing_for_finished_goods_comment?: string;
  follows_sampling_plans?: string;
  follows_sampling_plans_comment?: string;
  every_batch_tested_as_spec?: string;
  every_batch_tested_as_spec_comment?: string;
  retains_finished_goods_samples?: string;
  retains_finished_goods_samples_comment?: string;
  retention_years?: string;
  retention_years_comment?: string | number;
  personnel_trained?: string;
  personnel_trained_comment?: string;
  training_effective?: string;
  training_effective_comment?: string;
  training_records_maintained?: string;
  training_records_maintained_comment?: string;
  identification_of_non_conforming_material?: string;
  identification_of_non_conforming_material_comment?: string;
  investigation_for_non_conformance?: string;
  investigation_for_non_conformance_comment?: string;
  disposal_of_non_conformance?: string;
  disposal_of_non_conformance_comment?: string;
  non_conformance_records_available?: string;
  non_conformance_records_available_comment?: string;
  corrective_preventive_actions_taken?: string;
  corrective_preventive_actions_taken_comment?: string;
  handles_customer_complaints?: string;
  handles_customer_complaints_comment?: string;
  complaint_records_maintained?: string;
  complaint_records_maintained_comment?: string;
  corrective_action_effective?: string;
  corrective_action_effective_comment?: string;
  communicates_corrective_action_to_customers?: string;
  communicates_corrective_action_to_customers_comment?: string;
  feedback_system_from_customers?: string;
  feedback_system_from_customers_comment?: string;
  finished_goods_storage_method?: string;
  finished_goods_storage_method_comment?: string;
  follows_special_storage_conditions?: string;
  follows_special_storage_conditions_comment?: string;
  storage_condition_records_available?: string;
  storage_condition_records_available_comment?: string;
  assigns_shelf_life?: string;
  assigns_shelf_life_comment?: string;
  legislative_requirement_for_shipping?: string;
  legislative_requirement_for_shipping_comment?: string;
  mode_of_shipping?: string;
  mode_of_shipping_comment?: string;
  special_packaging_followed?: string;
  special_packaging_followed_comment?: string;
  shipping_controls?: string;
  shipping_controls_comment?: string;
  hazardous_or_non_hazardous: string;
  hazardous_or_non_hazardous_comment?: string;

  assement_status: string;
  created_at?: string;
  updated_at?: string;
  created_by_id?: string;
  created_by_name?: string;
  updated_by_id?: string | string[];
  updated_by_name?: string | string[];
}

// Auth headers (your original)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      Accept: "application/json",
      "Content-Type": "multipart/form-data",
    },
  };
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE SUPPLIER ASSESSMENT
export const createSupplierAssessment = async (
  data: SupplierAssessment
): Promise<{ Message: string; Detail: SupplierAssessment }> => {
  try {
    const formData = new FormData();
    Object.entries(data).forEach(([key, value]) => {
      if (value !== undefined && value !== null) {
        formData.append(key, String(value));
      }
    });

    const response = await axios.post(
      `${API_BASE_URL}/create-supplier-assessment`,
      formData,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ASSESSMENTS BY SUPPLIER ID
export const getSupplierAssessmentsBySupplierId = async (
  supplierId: string
): Promise<SupplierAssessment[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-supplier-assessments-by-supplier-id/${supplierId}`,
      {
        headers: {
          Authorization: localStorage.getItem("authToken") || "",
          Accept: "application/json",
        },
      }
    );
    return response.data.Data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL SUPPLIER ASSESSMENTS
export const getAllSupplierAssessments = async (): Promise<
  SupplierAssessment[]
> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-supplier-assessments`,
      {
        headers: {
          Authorization: localStorage.getItem("authToken") || "",
          Accept: "application/json",
        },
      }
    );
    return response.data.Details;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE SUPPLIER ASSESSMENT
export const updateSupplierAssessment = async (
  supplierId: string,
  data: Partial<SupplierAssessment>
): Promise<{ Message: string; Detail: SupplierAssessment }> => {
  try {
    const formData = new FormData();
    Object.entries(data).forEach(([key, value]) => {
      if (value !== undefined && value !== null) {
        formData.append(key, String(value));
      }
    });

    const response = await axios.put(
      `${API_BASE_URL}/update-supplier-assessment/${supplierId}`,
      formData,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE SUPPLIER ASSESSMENTS BY SUPPLIER ID
export const deleteSupplierAssessmentsBySupplierId = async (
  supplierId: string
): Promise<{ Message: string; DeletedCount: number }> => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-supplier-assessments-by-supplier-id/${supplierId}`,
      {
        headers: {
          Authorization: localStorage.getItem("authToken") || "",
          Accept: "application/json",
        },
      }
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
