// lib/api/supplier-test.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Interface (exactly as you wrote – NOT touched)
export interface SupplierTest {
  supplier_id: string;
  supplier_name?: string;
  product_id: string;
  product_name?: string;
  poNumber: string;
  poDate: string;
  deliveryDate: string;
  receivingDate: string;
  inspectionResult: string;
  rejectionReason: string;
  testedBy: string;
  testDate: string;
  testMethod: string;
  sampleBatchNo: string;
  testResultDetails: string;
  samplequantity?: string;
  status: string;
  testReportFile?: File | string | null;
  created_at?: string;
  updated_at?: string;
  created_by_id?: string;
  created_by_name?: string;
  updated_by_id?: string | null;
  updated_by_name?: string | null;
}

const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      Accept: "application/json",
    },
  };
};

const getMultipartHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      Accept: "application/json",
      "Content-Type": "multipart/form-data",
    },
  };
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE SUPPLIER TEST
export const createSupplierTest = async (data: SupplierTest) => {
  try {
    const formData = new FormData();

    if (data.testReportFile) {
      formData.append("testReportFile", data.testReportFile);
    }

    formData.append("testDate", data.testDate);
    formData.append("deliveryDate", data.deliveryDate);
    formData.append("supplierId", data.supplier_id);
    formData.append("testedBy", data.testedBy);
    formData.append("poNumber", data.poNumber);
    formData.append("product_id", data.product_id);
    formData.append("poDate", data.poDate);
    formData.append("rejectionReason", data.rejectionReason);
    formData.append("receivingDate", data.receivingDate);
    formData.append("sampleBatchNo", data.sampleBatchNo);
    formData.append("inspectionResult", data.inspectionResult);
    formData.append("Status", data.status);
    formData.append("testMethod", data.testMethod);
    formData.append("testResultDetails", data.testResultDetails);
    if (data.samplequantity) {
      formData.append("samplequantity", data.samplequantity);
    }

    const response = await axios.post(
      `${API_BASE_URL}/create-suppliertest`,
      formData,
      getMultipartHeaders()
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL SUPPLIER TESTS
export const getAllSupplierTests = async (): Promise<SupplierTest[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-all-suppliertests`,
      getAuthHeaders()
    );
    return response.data.DATA;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET SUPPLIER TESTS BY SUPPLIER ID
export const getSupplierTestsBySupplierId = async (
  supplierId: string
): Promise<SupplierTest[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-suppliertests-by-supplier-id/${supplierId}`,
      getAuthHeaders()
    );
    return response.data.DATA;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET SUPPLIER TEST BY SUPPLIER ID + PRODUCT ID
export const getSupplierTestByIds = async (
  supplierId: string,
  productId: string
): Promise<SupplierTest> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-suppliertests-by-supplier-id-product-id/${supplierId}-${productId}`,
      getAuthHeaders()
    );
    return response.data.DATA;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DOWNLOAD SUPPLIER TEST DOCUMENT
export const downloadSupplierTestFile = async (
  supplierId: string,
  productId: string
) => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/download-document-suppliertest/${supplierId}-${productId}`,
      {
        ...getAuthHeaders(),
        responseType: "blob",
      }
    );

    // Auto download with proper filename
    const contentDisposition = response.headers["content-disposition"];
    let fileName = `TestReport_${supplierId}_${productId}.pdf`;
    if (contentDisposition) {
      const match = contentDisposition.match(/filename="(.+?)"/);
      if (match?.[1]) fileName = match[1];
    }

    const url = window.URL.createObjectURL(new Blob([response.data]));
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", fileName);
    document.body.appendChild(link);
    link.click();
    link.remove();

    return response;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE SUPPLIER TEST
export const updateSupplierTest = async (
  supplierId: string,
  productId: string,
  updateFields: Partial<SupplierTest>
) => {
  try {
    const formData = new FormData();

    if (updateFields.inspectionResult)
      formData.append("inspectionResult", updateFields.inspectionResult);
    if (updateFields.status) formData.append("Status", updateFields.status);
    if (updateFields.testMethod)
      formData.append("testMethod", updateFields.testMethod);
    if (
      updateFields.testReportFile &&
      updateFields.testReportFile instanceof File
    )
      formData.append("testReportFile", updateFields.testReportFile);

    if (updateFields.poNumber)
      formData.append("poNumber", updateFields.poNumber);
    if (updateFields.poDate) formData.append("poDate", updateFields.poDate);
    if (updateFields.deliveryDate)
      formData.append("deliveryDate", updateFields.deliveryDate);
    if (updateFields.receivingDate)
      formData.append("receivingDate", updateFields.receivingDate);
    if (updateFields.testDate)
      formData.append("testDate", updateFields.testDate);
    if (updateFields.testedBy)
      formData.append("testedBy", updateFields.testedBy);
    if (updateFields.sampleBatchNo)
      formData.append("sampleBatchNo", updateFields.sampleBatchNo);
    if (updateFields.testResultDetails)
      formData.append("testResultDetails", updateFields.testResultDetails);
    if (updateFields.rejectionReason)
      formData.append("rejectionReason", updateFields.rejectionReason);
    if (updateFields.samplequantity)
      formData.append("samplequantity", updateFields.samplequantity);

    const response = await axios.put(
      `${API_BASE_URL}/update-suppliertest/${supplierId}-${productId}`,
      formData,
      getMultipartHeaders()
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE SUPPLIER TEST
export const deleteSupplierTest = async (
  supplierId: string,
  productId: string
) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-suppliertest/${supplierId}-${productId}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
