// lib/api/competency-evaluation.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

// Interface (exactly as you wrote – NOT touched)
export interface CompetencyEvaluation {
  evaluation_code?: string;
  employee_code: string;
  employee_name?: string;
  evaluation_date: string;
  competency_area: string;
  evaluation_method?: string;
  score?: string;
  competency_level?: "Beginner" | "Intermediate" | "Advanced";
  result?: string;
  remarks?: string;
  next_review_date?: string | null;
  evaluated_by_id?: string;
  evaluated_by_name?: string | null;
  approved_by_user_id?: string | null;
  approved_by_name?: string | null;
  approved_status?: string;
  created_at?: string;
  updated_at?: string;
  rejection_reason?: string;
  comments?: string | null;
  approved_date?: string | null;
}

// Auth Helper (your original – untouched)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  const authHeader = token.startsWith("Bearer ") ? token : `Bearer ${token}`;
  return {
    headers: {
      Authorization: authHeader,
      Accept: "application/json",
    },
  };
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.response?.data?.error ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE COMPETENCY EVALUATION
export const createCompetencyEvaluation = async (
  data: CompetencyEvaluation
) => {
  try {
    if (!data.employee_code || !data.evaluation_date || !data.competency_area) {
      throw new Error(
        "Missing required fields: employee_code, evaluation_date, competency_area"
      );
    }

    const form = new FormData();

    Object.entries(data).forEach(([key, value]) => {
      if (value !== undefined && value !== null && value !== "") {
        form.append(key, String(value));
      }
    });

    const response = await axios.post(
      `${API_BASE_PATH}/create_competency_evaluation`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL COMPETENCY EVALUATIONS
export const getAllCompetencyEvaluations = async (): Promise<
  CompetencyEvaluation[]
> => {
  try {
    const response = await axios.get(
      `${API_BASE_PATH}/get_all_evaluation`,
      getAuthHeaders()
    );

    if (
      response.data?.status === "success" &&
      Array.isArray(response.data.data)
    ) {
      return response.data.data;
    }

    return [];
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET SINGLE COMPETENCY EVALUATION BY CODE
export const getCompetencyEvaluationByCode = async (
  evaluation_code: string
): Promise<CompetencyEvaluation> => {
  try {
    const response = await axios.get(
      `${API_BASE_PATH}/get_by_evaluation/${evaluation_code}`,
      getAuthHeaders()
    );

    if (response.data?.status === "success" && response.data.data) {
      return response.data.data;
    }

    throw new Error("No competency evaluation data returned from API");
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE - THIS WAS MISSING TOAST! NOW FIXED
export const updateCompetencyEvaluation = async (
  evaluation_code: string,
  data: Partial<CompetencyEvaluation>
) => {
  try {
    const form = new FormData();
    Object.entries(data).forEach(([key, value]) => {
      if (value !== undefined && value !== null && value !== "") {
        form.append(key, String(value));
      }
    });

    const response = await axios.put(
      `${API_BASE_PATH}/update_competency/${evaluation_code}`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    toast.success(response.data.message || "Evaluation updated successfully"); // ← THIS WAS MISSING!
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE EVALUATION RESULT (APPROVAL SECTION)
export const updateEvaluationResult = async (
  evaluation_code: string,
  data: {
    approved_status?: string;
    rejection_reason?: string;
    comments?: string | null;
    result?: string;
    approved_by_user_id?: string | null;
    approved_date?: string | null;
  }
) => {
  try {
    if (!evaluation_code) {
      throw new Error("Missing required field: evaluation_code");
    }

    const form = new FormData();

    Object.entries(data).forEach(([key, value]) => {
      if (value !== undefined && value !== null && value !== "") {
        form.append(key, String(value));
      }
    });

    const response = await axios.put(
      `${API_BASE_PATH}/update_evaluation_result/${evaluation_code}`,
      form,
      {
        headers: {
          ...getAuthHeaders().headers,
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE COMPETENCY EVALUATION
export const deleteCompetencyEvaluation = async (evaluation_code: string) => {
  try {
    if (!evaluation_code) {
      throw new Error("Missing required field: evaluation_code");
    }

    const response = await axios.delete(
      `${API_BASE_PATH}/delete_competency/${evaluation_code}`,
      getAuthHeaders()
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
