// app/api/trainingsession.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Interface for Training Session
export interface TrainingSession {
  training_code: string;
  session_date: string; // YYYY-MM-DD
  duration_hours: string;
  trainer_id: string;
  venue?: string;
  remarks?: string;
}

// Interface for sessions returned by GET /all_sessions
export interface TrainingSessionRecord {
  session_id?: number;
  session_code: string;
  training_code: string;
  employee_code: string | null;
  employee_name: string | null;
  employee_department: string | null;
  employee_designation: string | null;
  topic: string | null;
  trainer_id: string;
  trainer_name: string | null;
  session_date: string;
  duration_hours: string;
  venue: string | null;
  remarks: string | null;
  status: string;
  rejection_reason: string | null;
  comments: string | null;
  trainer_sign_path: string | null;
  hr_sign_path: string | null;
  attendance_sheet_path: string | null;
  evidence_doc_path: string | null;
  employee_count: number;
  created_at: string;
  updated_at: string;
}

// Interface for upload response
export interface UploadSignatureResponse {
  status: string;
  message: string;
  trainer_signature: string;
  hr_signature: string;
  attendance_sheet: string;
  evidence_docs: string;
}

// Get auth headers
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      "Content-Type": "application/x-www-form-urlencoded",
      Accept: "application/json",
    },
  };
};

// Build form data
const buildTrainingForm = (data: TrainingSession): URLSearchParams => {
  return new URLSearchParams({
    training_code: data.training_code,
    session_date: data.session_date,
    duration_hours: data.duration_hours,
    trainer_id: data.trainer_id,
    venue: data.venue || "",
    remarks: data.remarks || "",
  });
};

// Show backend error
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE TRAINING SESSION
export const createTrainingSession = async (data: TrainingSession) => {
  try {
    const response = await axios.post(
      `${API_BASE_URL}/create_training_session`,
      buildTrainingForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL TRAINING SESSIONS
export const getAllTrainingSessions = async (): Promise<
  TrainingSessionRecord[]
> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/all_sessions`,
      getAuthHeaders()
    );
    return response.data.sessions;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET TRAINING SESSION BY CODE
export const getTrainingSessionByCode = async (
  session_code: string
): Promise<TrainingSessionRecord> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/training_session/${session_code}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE TRAINING SESSION
export const updateTrainingSession = async (
  session_code: string,
  data: TrainingSession
) => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update_training_session/${session_code}`,
      buildTrainingForm(data),
      getAuthHeaders()
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE TRAINING SESSION STATUS
export const updateTrainingSessionStatus = async (
  session_code: string,
  data: {
    status: string;
    rejection_reason?: string;
    comments?: string;
  }
) => {
  try {
    if (
      data.status === "Rejected" &&
      (!data.rejection_reason || data.rejection_reason.trim() === "")
    ) {
      toast.error("Rejection reason is required when rejecting a session.");
      return;
    }

    const form = new URLSearchParams({
      status: data.status,
      rejection_reason: data.rejection_reason || "",
      comments: data.comments || "",
    });

    const response = await axios.put(
      `${API_BASE_URL}/update_training_session_status/${session_code}`,
      form,
      getAuthHeaders()
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// Upload signatures (multipart/form-data)
export const uploadTrainingSessionSignatures = async (
  session_code: string,
  files: {
    trainer_sign?: File | null;
    hr_sign?: File | null;
    attendance_sheet?: File | null;
    evidence_doc?: File | null;
  }
): Promise<UploadSignatureResponse> => {
  try {
    const formData = new FormData();

    if (files.trainer_sign) formData.append("trainer_sign", files.trainer_sign);
    if (files.hr_sign) formData.append("hr_sign", files.hr_sign);
    if (files.attendance_sheet)
      formData.append("attendance_sheet", files.attendance_sheet);
    if (files.evidence_doc) formData.append("evidence_doc", files.evidence_doc);

    const token = localStorage.getItem("authToken") || "";

    const response = await axios.put(
      `${API_BASE_URL}/upload_signatures/${session_code}`,
      formData,
      {
        headers: {
          Authorization: token,
          Accept: "application/json",
          "Content-Type": "multipart/form-data",
        },
      }
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE TRAINING SESSION
export const deleteTrainingSession = async (session_code: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete_training_session/${session_code}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// CREATE ATTENDANCE
export const createAttendance = async (data: {
  session_code: string;
  session_date: string;
  attended: string;
  remarks?: string;
}) => {
  try {
    const form = new URLSearchParams({
      session_code: data.session_code,
      session_date: data.session_date,
      attended: data.attended,
      remarks: data.remarks || "",
    });

    const response = await axios.post(
      `${API_BASE_URL}/create_attendance`,
      form,
      getAuthHeaders()
    );

    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL ATTENDANCE
export interface AttendanceRecord {
  attendance_code: string;
  attended: string;
  score: string | null;
  certification_status: string | null;
  remarks: string | null;
  created_at: string;
  updated_at: string;
  session_code: string;
  session_date: string;
  training_topic: string;
  trainer_id: string;
  venue: string;
  employee_name: string;
}

export const getAllAttendance = async (): Promise<AttendanceRecord[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get_all_attendance`,
      getAuthHeaders()
    );

    return response.data.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// ✅ NEW — GET ATTENDANCE BY CODE
export const getAttendanceByCode = async (
  attendance_code: string
): Promise<AttendanceRecord> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get_by_attendance/${attendance_code}`,
      getAuthHeaders()
    );

    return response.data.data; // backend returns { status, data: {...} }
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// ✅ NEW — UPDATE ATTENDANCE
export const updateAttendance = async (
  attendance_code: string,
  data: {
    score?: number | null;
    certification_status?: string | null;
    remarks?: string | null;
  }
) => {
  try {
    const form = new URLSearchParams({
      score: data.score ? String(data.score) : "",
      certification_status: data.certification_status || "",
      remarks: data.remarks || "",
    });

    const response = await axios.put(
      `${API_BASE_URL}/update_attendance/${attendance_code}`,
      form,
      getAuthHeaders()
    );

    return response.data; // { status, message, attendance_code }
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// ✅ NEW — DELETE ATTENDANCE
export const deleteAttendance = async (attendance_code: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/attendance/delete/${attendance_code}`,
      getAuthHeaders()
    );
    return response.data; // { status, message, attendance_code }
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
