'use client';
import { useState, useRef } from 'react';
import FullCalendar from '@fullcalendar/react';
import { EventClickArg, EventContentArg, EventInput } from '@fullcalendar/core';
import dayGridPlugin from '@fullcalendar/daygrid';
import timeGridPlugin from '@fullcalendar/timegrid';
import interactionPlugin from '@fullcalendar/interaction';
import { Dialog } from '@headlessui/react';
import { Plus, X, CalendarDaysIcon } from 'lucide-react';
import toast from 'react-hot-toast';

type EventType = 'Audit' | 'Task' | 'Reminder';
type EventUser = 'Alice' | 'Bob';

interface EventFormData extends EventInput {
  id?: string;
  title?: string;
  start?: string;
  type?: EventType;
  user?: EventUser;
}

export default function CalendarPage() {
  const [events, setEvents] = useState<EventFormData[]>([
    { id: '1', title: 'Audit Review', start: '2025-08-05', type: 'Audit', user: 'Alice' },
    { id: '2', title: 'Calibration Task', start: '2025-08-10', type: 'Task', user: 'Bob' },
    { id: '3', title: 'Reminder: ISO', start: '2025-08-15', type: 'Reminder', user: 'Alice' },
  ]);
  const [filterType, setFilterType] = useState<'All' | EventType>('All');
  const [filterUser, setFilterUser] = useState<'All' | EventUser>('All');
  const [modalOpen, setModalOpen] = useState(false);
  const formRef = useRef<EventFormData | null>(null);

  const filtered = events.filter(e =>
    (filterType === 'All' || e.type === filterType) &&
    (filterUser === 'All' || e.user === filterUser)
  );

  const handleDateClick = (arg: { dateStr: string }) => {
    formRef.current = { start: arg.dateStr };
    setModalOpen(true);
  };

  const handleEventClick = (arg: EventClickArg) => {
    formRef.current = {
      id: arg.event.id,
      title: arg.event.title,
      start: arg.event.startStr,
      type: arg.event.extendedProps.type,
      user: arg.event.extendedProps.user,
    };
    setModalOpen(true);
  };

  const saveEvent = () => {
    const f = formRef.current;
    if (!f) return;

    if (f.id) {
      setEvents(prev => prev.map(e => e.id === f.id ? f : e));
      toast.success('Event updated');
    } else {
      setEvents(prev => [...prev, { ...f, id: Date.now().toString() }]);
      toast.success('Event added');
    }
    setModalOpen(false);
  };

  const renderEventContent = (eventInfo: EventContentArg) => {
    return (
      <div className="flex items-center gap-1 text-xs">
        <span
          className={`w-2 h-2 rounded-full ${
            eventInfo.event.extendedProps.type === 'Audit'
              ? 'bg-blue-500'
              : eventInfo.event.extendedProps.type === 'Task'
              ? 'bg-green-500'
              : 'bg-yellow-500'
          }`}
        />
        <b>{eventInfo.event.title}</b>
      </div>
    );
  };

  return (
    <main className="w-full min-h-screen bg-gray-50 dark:bg-[#0f172a] transition-colors px-4 py-6">
      <div className="flex flex-wrap items-center justify-between mb-6 gap-4">
  <div>
    <h1 className="flex items-center gap-2 text-3xl font-bold text-gray-800 dark:text-white">
  <CalendarDaysIcon className="w-8 h-8 text-green-600 dark:text-green-400 flex-shrink-0" />
  <span>Calendar</span>
</h1>
    <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
      This calendar helps you schedule and monitor quality-related activities.
    </p>
  </div>
  <button
    onClick={() => {
      formRef.current = {};
      setModalOpen(true);
    }}
    className="flex items-center gap-2 bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg shadow whitespace-nowrap"
  >
    <Plus className="w-5 h-5" /> Add Event
  </button>
</div>

      <div className="flex flex-wrap gap-4 mb-4">
        <select
          value={filterType}
          onChange={e => setFilterType(e.target.value as EventType | 'All')}
          className="px-3 py-2 border rounded-lg bg-white dark:bg-gray-700 border-gray-300 dark:border-gray-600 text-gray-800 dark:text-white"
        >
          <option>All</option><option>Audit</option><option>Task</option><option>Reminder</option>
        </select>
        <select
          value={filterUser}
          onChange={e => setFilterUser(e.target.value as EventUser | 'All')}
          className="px-3 py-2 border rounded-lg bg-white dark:bg-gray-700 border-gray-300 dark:border-gray-600 text-gray-800 dark:text-white"
        >
          <option>All</option><option>Alice</option><option>Bob</option>
        </select>
      </div>

      <div className="bg-white dark:bg-gray-800 rounded-xl shadow p-4">
        <FullCalendar
          plugins={[dayGridPlugin, timeGridPlugin, interactionPlugin]}
          initialView="dayGridMonth"
          events={filtered}
          dateClick={handleDateClick}
          eventClick={handleEventClick}
          height="auto"
          eventContent={renderEventContent}
          headerToolbar={{
            left: 'prev,next today',
            center: 'title',
            right: 'dayGridMonth,timeGridWeek,timeGridDay'
          }}
        />
      </div>

      <Dialog open={modalOpen} onClose={() => setModalOpen(false)} className="fixed inset-0 z-50 flex items-center justify-center">
        <div className="fixed inset-0 bg-black/30" aria-hidden="true" />
        <Dialog.Panel className="bg-white dark:bg-gray-800 rounded-lg p-6 max-w-md mx-auto z-10">
          <div className="flex justify-between items-center mb-4">
            <Dialog.Title className="text-xl font-semibold text-gray-800 dark:text-white">
              {formRef.current?.id ? 'Edit Event' : 'New Event'}
            </Dialog.Title>
            <button onClick={() => setModalOpen(false)}>
              <X className="w-6 h-6 text-gray-500 hover:text-gray-700 dark:text-gray-400" />
            </button>
          </div>
          <div className="space-y-3">
            <input
              type="text"
              defaultValue={formRef.current?.title || ''}
              placeholder="Title"
              onChange={e => { if (formRef.current) formRef.current.title = e.target.value }}
              className="w-full px-3 py-2 border rounded-lg bg-gray-100 dark:bg-gray-700 border-gray-300 dark:border-gray-600"
            />
            <input
              type="date"
              defaultValue={formRef.current?.start || ''}
              onChange={e => { if (formRef.current) formRef.current.start = e.target.value }}
              className="w-full px-3 py-2 border rounded-lg bg-gray-100 dark:bg-gray-700 border-gray-300 dark:border-gray-600"
            />
            <select
              defaultValue={formRef.current?.type || 'Task'}
              onChange={e => { if (formRef.current) formRef.current.type = e.target.value as EventType }}
              className="w-full px-3 py-2 border rounded-lg bg-gray-100 dark:bg-gray-700 border-gray-300 dark:border-gray-600"
            >
              <option>Audit</option><option>Task</option><option>Reminder</option>
            </select>
            <select
              defaultValue={formRef.current?.user || 'Alice'}
              onChange={e => { if (formRef.current) formRef.current.user = e.target.value as EventUser }}
              className="w-full px-3 py-2 border rounded-lg bg-gray-100 dark:bg-gray-700 border-gray-300 dark:border-gray-600"
            >
              <option>Alice</option><option>Bob</option>
            </select>
            <button
              onClick={saveEvent}
              className="w-full bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg"
            >Save</button>
          </div>
        </Dialog.Panel>
      </Dialog>
    </main>
  );
}
