"use client";

import { usePathname, useRouter } from "next/navigation";
import Sidebar from "./Sidebar";
import Navbar from "./Navbar";
import { useState, useEffect } from "react";
import { motion, AnimatePresence } from "framer-motion";
import Footer from "./Footer";
import { CircularLoader, DotsLoader } from "./common/AppLoader";
import { useAuth } from "@/context/AuthContext";

export default function LayoutWrapper({
  children,
}: {
  children: React.ReactNode;
}) {
  const pathname = usePathname();
  const router = useRouter();
  const noLayoutRoutes = [
    "/login",
    "/login/forgotpassword",
    "/login/resetpassword",
  ];
  const isAuthPage = noLayoutRoutes.includes(pathname);

  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const [localLoading, setLocalLoading] = useState(true);

  const { user, loading } = useAuth();

  useEffect(() => {
    setIsMobileMenuOpen(false);
  }, [pathname]);

  useEffect(() => {
    const timeout = setTimeout(() => setLocalLoading(false), 800);
    return () => clearTimeout(timeout);
  }, []);

  // Redirect if not logged in and not on login page
  useEffect(() => {
    if (!loading && !user && !isAuthPage) {
      router.replace("/login");
    }
  }, [loading, user, isAuthPage, router]);

  // While auth is loading, show loader
  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <CircularLoader />
      </div>
    );
  }

  // If on login/forgot/reset, just show the page without layout
  if (isAuthPage) return <>{children}</>;

  return (
    <>
      {/* {localLoading && <DotsLoader />} */}
      <div className="flex h-screen bg-gray-50 dark:bg-gray-900 transition-colors duration-300 overflow-hidden">
        <AnimatePresence>
          {isMobileMenuOpen && (
            <motion.div
              initial={{ x: "-100%" }}
              animate={{ x: 0 }}
              exit={{ x: "-100%" }}
              transition={{ type: "spring", stiffness: 300, damping: 30 }}
              className="lg:hidden fixed inset-y-0 left-0 z-50"
            >
              <Sidebar onMobileToggle={() => setIsMobileMenuOpen(false)} />
            </motion.div>
          )}
        </AnimatePresence>

        <div className="hidden lg:block">
          <Sidebar />
        </div>

        <div className="flex flex-col flex-1 min-w-0">
          <Navbar
            onMobileToggle={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
          />
          <main className="flex-1 min-h-0 overflow-y-auto p-2 sm:p-2 lg:p-3 md:p-3">
            {children}
          </main>
          <Footer />
        </div>

        {isMobileMenuOpen && (
          <div
            className="lg:hidden fixed inset-0 bg-black/50 z-40"
            onClick={() => setIsMobileMenuOpen(false)}
          />
        )}
      </div>
    </>
  );
}
