"use client";

import React, { useMemo } from "react";
import Table  from "antd/es/table";
import type { ColumnsType , TableProps } from "antd/es/table";

export interface DataTableProps<T> {
  columns: ColumnsType<T>;
  dataSource: T[];
  loading?: boolean;
  scrollX?: number | string | { x: string | number };
  pagination?: TableProps<T>["pagination"];
  rowKey?: string | ((record: T) => string);
  bordered?: boolean;
  onRowClick?: (record: T) => void;
  sticky?: boolean;
  className?: string;
}

function DataTableComponent<T extends object>({
  columns,
  dataSource,
  loading = false,
  scrollX = "max-content",
  pagination = { pageSize: 10, showSizeChanger: true },
  rowKey = "id",
  bordered = true,
  onRowClick,
  sticky = false,
  className,
}: DataTableProps<T>) {
  const scrollValue = useMemo(
    () => (typeof scrollX === "object" ? scrollX : { x: scrollX }),
    [scrollX]
  );

return loading ? (
  <div className="space-y-2">
    {Array.from({ length: 5 }).map((_, i) => (
      <div
        key={i}
        className="h-6 bg-green-100 dark:bg-green-700 rounded animate-pulse"
      />
    ))}
  </div>
) : (
    <div className="rounded-2xl border border-gray-200 shadow-sm overflow-hidden dark:border-gray-800 dark:bg-gray-900">
      <Table<T>
        columns={columns}
        dataSource={dataSource}
        scroll={scrollValue}
        pagination={pagination}
        rowKey={rowKey}
        bordered={bordered}
        sticky={sticky}
        onRow={
          onRowClick
            ? (record) => ({ onClick: () => onRowClick(record) })
            : undefined
        }
        className={`
          [&_.ant-table]:!bg-transparent
          [&_.ant-table-thead>tr>th]:!bg-gray-100 dark:[&_.ant-table-thead>tr>th]:!bg-gray-800 
          [&_.ant-table-thead>tr>th]:!text-gray-700 dark:[&_.ant-table-thead>tr>th]:!text-gray-300 
          [&_.ant-table-thead>tr>th]:!font-semibold [&_.ant-table-thead>tr>th]:!text-sm
          [&_.ant-table-tbody>tr>td]:!text-gray-800 dark:[&_.ant-table-tbody>tr>td]:!text-gray-200
          [&_.ant-table-tbody>tr:hover>td]:!bg-gray-50 dark:[&_.ant-table-tbody>tr:hover>td]:!bg-gray-800
          [&_.ant-table-tbody>tr>td]:!text-sm [&_.ant-table-tbody>tr>td]:!align-middle
          [&_.ant-pagination]:!p-4
          [&_.ant-pagination-item]:!rounded-lg
          [&_.ant-pagination-item-active]:!bg-primary !text-white
          ${className || ""}
        `}
      />
    </div>
  );
}

const DataTable = React.memo(DataTableComponent) as typeof DataTableComponent;

export default DataTable;
