"use client";

import React, { useState } from "react";
import { Button } from "antd";
import {
  Users,
  BarChart3,
  BookOpen,
  CalendarDays,
  CheckCircle,
  ClipboardList,
  FileText,
  GraduationCap,
  Layers,
  Smile,
  Star,
  TrendingUp,
  Menu,
} from "lucide-react";
import {
  ResponsiveContainer,
  PieChart,
  Pie,
  Cell,
  Tooltip,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Legend,
  AreaChart,
  Area,
} from "recharts";
import { motion } from "framer-motion";

const COLORS = ["#2563eb", "#16a34a", "#f59e0b", "#ef4444"];

const qualificationData = [
  { name: "Graduate", value: 60 },
  { name: "Post-Graduate", value: 25 },
  { name: "Diploma", value: 10 },
  { name: "Others", value: 5 },
];

const trainingData = [
  { name: "Completed", value: 65 },
  { name: "Pending", value: 25 },
  { name: "Overdue", value: 10 },
];

const performanceData = [
  { month: "Jan", score: 70 },
  { month: "Feb", score: 75 },
  { month: "Mar", score: 80 },
  { month: "Apr", score: 78 },
  { month: "May", score: 85 },
  { month: "Jun", score: 90 },
];

export default function HRMDashboard() {
  const [sidebarOpen, setSidebarOpen] = useState(false);

  return (
    <div className="flex min-h-screen bg-gray-50">
      <main className="flex-1 w-full px-4 sm:px-6 py-4 sm:py-6">
        <h1 className="flex items-center gap-3 text-2xl sm:text-3xl font-bold text-gray-800 mb-4 sm:mb-6">
          <Users className="w-7 h-7 text-green-600 flex-shrink-0" />
          <span className="text-green-600">Human Resources Dashboard</span>
        </h1>
        <p className="text-gray-600 text-base sm:text-lg mb-4 sm:mb-6">
          Empowering workforce performance through intelligent analytics and MDQMS compliance.
        </p>

        {/* ✅ Main Dashboard Content */}
        <div
          className={`transition-all duration-500 ${
            sidebarOpen ? "opacity-30 pointer-events-none" : "opacity-100"
          }`}
        >
          <main className="space-y-10">
            {/* Summary Cards */}
            <motion.div
              initial="hidden"
              animate="visible"
              variants={{
                hidden: {},
                visible: { transition: { staggerChildren: 0.15 } },
              }}
              className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6"
            >
              <SummaryCard
                title="Total Employees"
                value="235"
                icon={<Users className="text-blue-500" />}
              />
              <SummaryCard
                title="Active Trainings"
                value="12"
                icon={<GraduationCap className="text-green-500" />}
              />
              <SummaryCard
                title="Pending Appraisals"
                value="8"
                icon={<TrendingUp className="text-amber-500" />}
              />
              <SummaryCard
                title="HR Announcements"
                value="3"
                icon={<Smile className="text-purple-500" />}
              />
            </motion.div>

            {/* Charts Section */}
            <motion.div
              initial={{ opacity: 0, y: 40 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.7 }}
              viewport={{ once: true }}
              className="grid grid-cols-1 lg:grid-cols-2 gap-8"
            >
              <ChartCard
                title="Employee Qualification Overview"
                icon={<ClipboardList className="text-green-600" />}
              >
                <ResponsiveContainer width="100%" height={320}>
                  <BarChart data={qualificationData} barSize={40}>
                    <defs>
                      <linearGradient id="qualBar" x1="0" y1="0" x2="0" y2="1">
                        <stop
                          offset="5%"
                          stopColor="#22c55e"
                          stopOpacity={0.9}
                        />
                        <stop
                          offset="95%"
                          stopColor="#86efac"
                          stopOpacity={0.3}
                        />
                      </linearGradient>
                    </defs>
                    <CartesianGrid strokeDasharray="3 3" stroke="#e5e7eb" />
                    <XAxis
                      dataKey="name"
                      stroke="#6b7280"
                      tick={{ fontSize: 12 }}
                    />
                    <YAxis stroke="#6b7280" />
                    <Tooltip
                      contentStyle={{
                        borderRadius: "10px",
                        backgroundColor: "#fff",
                      }}
                    />
                    <Legend />
                    <Bar
                      dataKey="value"
                      fill="url(#qualBar)"
                      radius={[8, 8, 0, 0]}
                    />
                  </BarChart>
                </ResponsiveContainer>
              </ChartCard>

              <ChartCard
                title="Training Completion Status"
                icon={<BookOpen className="text-green-600" />}
              >
                <ResponsiveContainer width="100%" height={300}>
                  <PieChart>
                    <Pie
                      data={trainingData}
                      startAngle={180}
                      endAngle={0}
                      cx="50%"
                      cy="80%"
                      outerRadius={120}
                      fill="#8884d8"
                      dataKey="value"
                    >
                      {trainingData.map((_, index) => (
                        <Cell
                          key={`cell-${index}`}
                          fill={COLORS[index % COLORS.length]}
                        />
                      ))}
                    </Pie>
                    <Tooltip
                      contentStyle={{
                        borderRadius: "10px",
                        backgroundColor: "#fff",
                      }}
                    />
                    <Legend />
                  </PieChart>
                </ResponsiveContainer>
              </ChartCard>
            </motion.div>

            {/* Gradient Area Chart */}
            <motion.div
              initial={{ opacity: 0, y: 40 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.6 }}
              viewport={{ once: true }}
            >
              <ChartCard
                title="Average Performance Scores (Last 6 Months)"
                icon={<BarChart3 className="text-blue-600" />}
              >
                <ResponsiveContainer width="100%" height={320}>
                  <AreaChart data={performanceData}>
                    <defs>
                      <linearGradient
                        id="colorScore"
                        x1="0"
                        y1="0"
                        x2="0"
                        y2="1"
                      >
                        <stop
                          offset="5%"
                          stopColor="#3b82f6"
                          stopOpacity={0.8}
                        />
                        <stop
                          offset="95%"
                          stopColor="#3b82f6"
                          stopOpacity={0.1}
                        />
                      </linearGradient>
                    </defs>
                    <CartesianGrid strokeDasharray="3 3" stroke="#e5e7eb" />
                    <XAxis dataKey="month" stroke="#6b7280" />
                    <YAxis stroke="#6b7280" />
                    <Tooltip
                      contentStyle={{
                        borderRadius: "10px",
                        backgroundColor: "#fff",
                      }}
                    />
                    <Area
                      type="monotone"
                      dataKey="score"
                      stroke="#3b82f6"
                      fill="url(#colorScore)"
                      strokeWidth={3}
                      dot={{ r: 5, stroke: "#2563eb", strokeWidth: 2 }}
                    />
                  </AreaChart>
                </ResponsiveContainer>
              </ChartCard>
            </motion.div>

            {/* HR Modules */}
            <motion.div
              initial="hidden"
              whileInView="visible"
              viewport={{ once: true }}
              variants={{
                hidden: {},
                visible: { transition: { staggerChildren: 0.1 } },
              }}
              className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6"
            >
              <ModuleCard
                icon={<Users />}
                title="Employee Master"
                desc="Manage employee profiles"
              />
              <ModuleCard
                icon={<ClipboardList />}
                title="Minimum Qualification"
                desc="Qualification benchmarks"
              />
              <ModuleCard
                icon={<Layers />}
                title="Competency & Skill Matrix"
                desc="Evaluate employee skills"
              />
              <ModuleCard
                icon={<BookOpen />}
                title="Training Need Identification"
                desc="Analyze learning needs"
              />
              <ModuleCard
                icon={<CalendarDays />}
                title="Annual Training Schedule"
                desc="Plan sessions"
              />
              <ModuleCard
                icon={<FileText />}
                title="Training Record"
                desc="Track completed trainings"
              />
              <ModuleCard
                icon={<Star />}
                title="Training Record Card"
                desc="Individual summaries"
              />
              <ModuleCard
                icon={<CheckCircle />}
                title="Evaluation"
                desc="Measure outcomes"
              />
            </motion.div>
          </main>
        </div>
      </main>
    </div>
  );
}

/* ---------- Reusable Components ---------- */

function SummaryCard({ title, value, icon }: any) {
  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.4 }}
      whileHover={{ scale: 1.03 }}
      className="rounded-2xl bg-white/90 backdrop-blur-md border border-gray-200 shadow-md hover:shadow-xl p-5 transition-all duration-300 w-full"
    >
      <div className="flex justify-between items-center">
        <div>
          <p className="text-sm text-gray-500">{title}</p>
          <h3 className="text-3xl font-bold">{value}</h3>
        </div>
        <div className="p-3 rounded-full bg-gradient-to-br from-gray-100 to-gray-200 flex items-center justify-center">
          {icon}
        </div>
      </div>
    </motion.div>
  );
}

function ChartCard({ title, icon, children }: any) {
  return (
    <motion.div
      initial={{ opacity: 0, y: 30 }}
      whileInView={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.5 }}
      viewport={{ once: true }}
      className="rounded-2xl bg-white/80 backdrop-blur-lg border border-gray-200 shadow-lg hover:shadow-2xl p-4 sm:p-6 transition-all w-full overflow-hidden"
    >
      <h2 className="text-lg font-semibold mb-4 flex items-center gap-2 text-gray-700 flex-wrap">
        {icon} {title}
      </h2>
      <div className="relative w-full">{children}</div>
    </motion.div>
  );
}

function ModuleCard({ icon, title, desc }: any) {
  return (
    <motion.div
      whileHover={{ scale: 1.03, y: -4 }}
      transition={{ duration: 0.3 }}
      className="rounded-2xl border border-gray-200 bg-white/80 backdrop-blur-md shadow-sm hover:shadow-lg transition-all p-5 flex flex-col gap-3 w-full"
    >
      <div className="w-12 h-12 flex items-center justify-center bg-green-100 text-green-700 rounded-full shadow-inner">
        {icon}
      </div>
      <h3 className="text-lg font-semibold break-words">{title}</h3>
      <p className="text-sm text-gray-500 flex-1">{desc}</p>
      <Button
        type="default"
        className="mt-auto border-green-300 text-green-600 hover:bg-green-50 hover:border-green-400 rounded-lg w-full sm:w-auto"
      >
        View
      </Button>
    </motion.div>
  );
}
