"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import Form from "antd/es/form";
import Input from "antd/es/input";
import Modal from "antd/es/modal";
import Select from "antd/es/select";
import Space from "antd/es/space";
import Tooltip from "antd/es/tooltip";
import Tag from "antd/es/tag";
import Breadcrumb from "antd/es/breadcrumb";
import Card from "antd/es/card";
import DatePicker from "antd/es/date-picker";
import Switch from "antd/es/switch";
import Pagination from "antd/es/pagination";
import Dropdown from "antd/es/dropdown";
import type { ColumnsType } from "antd/es/table";
import Table from "antd/es/table";
import {
  PlusOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleOutlined,
  StopOutlined,
  TeamOutlined,
  SearchOutlined,
  FileTextOutlined,
  CalendarOutlined,
  CheckCircleTwoTone,
  WarningOutlined,
  EditOutlined,
} from "@ant-design/icons";
import { FaFileExcel, FaFilePdf, FaTrash, FaFileExport } from "react-icons/fa";
import { LuCalendarDays } from "react-icons/lu";
import { toast } from "sonner";
import dayjs from "@/data/utils/dayjs";
import GlassCard from "../../components/common/GlassCard";
import WidgetCard from "../../components/common/WidgetCard";
import dynamic from "next/dynamic";
import ConfirmDialog from "../../components/common/ConfirmDialog";
import DataTable from "../../components/common/DataTable";
import {
  createAnnualSchedule,
  getAllAnnualSchedules,
  deleteAnnualSchedule,
  approveAnnualSchedule,
  getScheduleItems,
  addTrainingItem,
  updateTrainingItem,
  deleteTrainingItem,
  getAllTrainers,
} from "@/app/api/annualtrainingschedule/annualschedule";
import { getAllTrainingNeeds } from "@/app/api/trainingneedidentification/trainingneeds";

const ColumnVisibilityToggle = dynamic(
  () => import("../../components/common/ManageColumn"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
  }
);

const { RangePicker } = DatePicker;

interface TrainingItem {
  annual_training_item_id: string;
  training_id: string;
  training_code: string;
  training_topic: string;
  planned_month: string;
  planned_date?: string;
  trainer_id?: string;
  trainer_name?: string;
  remarks?: string;
  attendance_status?: "Not Started" | "Completed" | "Missed" | "Postponed";
}

interface AnnualSchedule {
  schedule_code: string;
  year: string;
  status: "Draft" | "Approved" | "Rejected" | null;
  created_at: string;
  created_by_name?: string;
  reviewed_by_name?: string;
  rejection_reason?: string;
  items?: TrainingItem[];
}

export default function AnnualTrainingSchedulePage() {
  const [form] = Form.useForm();
  const [createForm] = Form.useForm();
  const [approveForm] = Form.useForm();
  const [itemsForm] = Form.useForm();

  const [schedules, setSchedules] = useState<AnnualSchedule[]>([]);
  const [trainers, setTrainers] = useState<{ id: string; name: string }[]>([]);
  const [trainingNeeds, setTrainingNeeds] = useState<
    { id: string; code: string; topic: string; employee_name: string }[]
  >([]);
  const [loading, setLoading] = useState(false);
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [isGridView, setIsGridView] = useState(false);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [selectedStatus, setSelectedStatus] = useState<string | undefined>();

  // Modals
  const [createModalOpen, setCreateModalOpen] = useState(false);
  const [itemsModalOpen, setItemsModalOpen] = useState(false);
  const [selectedSchedule, setSelectedSchedule] =
    useState<AnnualSchedule | null>(null);
  const [editingItem, setEditingItem] = useState<TrainingItem | null>(null);

  // Approve/Reject Modal
  const [approveModalOpen, setApproveModalOpen] = useState(false);
  const [currentActionSchedule, setCurrentActionSchedule] =
    useState<AnnualSchedule | null>(null);
  const [actionType, setActionType] = useState<"Approved" | "Rejected">(
    "Approved"
  );

  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    title: "",
    message: "",
    onConfirm: () => {},
  });

  // Auto grid view on mobile
  useEffect(() => {
    const handleResize = () => {
      setIsGridView(window.innerWidth < 768);
    };
    handleResize();
    window.addEventListener("resize", handleResize);
    return () => window.removeEventListener("resize", handleResize);
  }, []);

  useEffect(() => {
    fetchSchedules();
    fetchTrainers();
    fetchTrainingNeeds();
  }, []);

  const fetchSchedules = async () => {
    setLoading(true);
    try {
      const data = await getAllAnnualSchedules();
      const normalized = (Array.isArray(data) ? data : []).map((s: any) => ({
        ...s,
        status: s.status ?? "Draft",
      }));
      setSchedules(
        normalized.sort(
          (a, b) => dayjs(b.created_at).unix() - dayjs(a.created_at).unix()
        )
      );
    } catch {
      // Error already toasted in API layer
    } finally {
      setLoading(false);
    }
  };

  const fetchTrainers = async () => {
    try {
      const data = await getAllTrainers();
      setTrainers(Array.isArray(data) ? data : []);
    } catch {
      // Error already toasted in API layer
    }
  };

  const fetchTrainingNeeds = async () => {
    try {
      const response: any = await getAllTrainingNeeds();
      let dataArray: any[] = [];
      if (response) {
        if (Array.isArray(response)) dataArray = response;
        else if (response.data && Array.isArray(response.data))
          dataArray = response.data;
        else if (response.Datas && Array.isArray(response.Datas))
          dataArray = response.Datas;
      }
      if (dataArray.length === 0) {
        toast.info("No training needs found");
        setTrainingNeeds([]);
        return;
      }
      const approvedNeeds = dataArray.filter(
        (item: any) => item.approval_status === "Approved" && item.employee_name
      );
      if (approvedNeeds.length === 0) {
        toast.info("No approved training needs available");
        setTrainingNeeds([]);
        return;
      }
      const formatted = approvedNeeds.map((item: any) => ({
        id: item.training_code,
        code: item.training_code,
        topic: item.training_topic || "No Topic",
        employee_name: item.employee_name || "Unknown Employee",
      }));
      setTrainingNeeds(formatted);
    } catch {
      // Error already toasted in API layer
      setTrainingNeeds([]);
    }
  };

  const openCreateModal = () => {
    createForm.resetFields();
    createForm.setFieldsValue({ year: dayjs().format("YYYY") });
    setCreateModalOpen(true);
  };

  const handleCreateSchedule = async () => {
    try {
      const { year } = await createForm.validateFields();
      const result = await createAnnualSchedule(year);
      toast.success(result?.message || "Annual schedule created successfully");
      setCreateModalOpen(false);
      fetchSchedules();
    } catch {
      // Validation or backend error → already shown in API layer
    }
  };

  const openItemsModal = async (schedule: AnnualSchedule) => {
    setSelectedSchedule(schedule);
    setItemsModalOpen(true);
    setEditingItem(null);
    itemsForm.resetFields();

    try {
      const res = await getScheduleItems(schedule.schedule_code);
      const items = Array.isArray(res?.data) ? res.data : [];
      setSelectedSchedule({ ...schedule, items });
    } catch {
      // Error already toasted in API layer
    }
  };

  const reloadScheduleItems = async (schedule_code: string) => {
    try {
      const res = await getScheduleItems(schedule_code);
      const items = Array.isArray(res?.data) ? res.data : [];
      setSelectedSchedule((prev) => (prev ? { ...prev, items } : prev));
    } catch {
      // Error already toasted in API layer
    }
  };

  const handleAddItem = async () => {
    if (!selectedSchedule || selectedSchedule.status !== "Draft") {
      toast.error("Cannot add items — schedule must be in Draft");
      return;
    }
    try {
      const values = await itemsForm.validateFields();
      const result = await addTrainingItem(selectedSchedule.schedule_code, {
        training_id: values.training_id,
        planned_month: values.planned_month,
        trainer_id: values.trainer_id || undefined,
        planned_date: values.planned_date?.format("YYYY-MM-DD") || undefined,
        remarks: values.remarks || undefined,
      });
      toast.success(result?.message || "Training item added");
      itemsForm.resetFields();
      await reloadScheduleItems(selectedSchedule.schedule_code);
    } catch {
      // Error already toasted in API layer
    }
  };

  const handleUpdateItem = async () => {
    if (!editingItem) return;
    try {
      const result = await updateTrainingItem(
        editingItem.annual_training_item_id,
        {
          planned_month: itemsForm.getFieldValue("planned_month"),
          planned_date: itemsForm
            .getFieldValue("planned_date")
            ?.format("YYYY-MM-DD"),
          trainer_id: itemsForm.getFieldValue("trainer_id") || undefined,
          remarks: itemsForm.getFieldValue("remarks") || undefined,
        }
      );
      toast.success(result?.message || "Item updated");
      setEditingItem(null);
      itemsForm.resetFields();
      await reloadScheduleItems(selectedSchedule!.schedule_code);
    } catch {
      // Error already toasted in API layer
    }
  };

  const handleDeleteSchedule = (code: string, status?: string | null) => {
    if (status === "Approved") {
      toast.error("Cannot delete approved schedule");
      return;
    }
    setConfirmDialog({
      open: true,
      title: "Delete Schedule",
      message: "This action cannot be undone.",
      onConfirm: async () => {
        try {
          await deleteAnnualSchedule(code);
          toast.success("Schedule deleted");
          fetchSchedules();
        } catch {
          // Error already toasted in API layer
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;
    const hasApproved = schedules
      .filter((s) => selectedRows.includes(s.schedule_code))
      .some((s) => s.status === "Approved");
    if (hasApproved) {
      toast.error("Cannot delete approved schedules");
      return;
    }
    setConfirmDialog({
      open: true,
      title: "Bulk Delete",
      message: `Delete ${selectedRows.length} schedule(s)?`,
      onConfirm: async () => {
        try {
          await Promise.all(selectedRows.map(deleteAnnualSchedule));
          toast.success("Schedules deleted");
          setSelectedRows([]);
          fetchSchedules();
        } catch {
          // Error already toasted in API layer
        }
      },
    });
  };

  const openApproveModal = (schedule: AnnualSchedule) => {
    setCurrentActionSchedule(schedule);
    setActionType("Approved");
    approveForm.resetFields();
    setApproveModalOpen(true);
  };

  const handleApproveSubmit = async () => {
    try {
      if (actionType === "Rejected") {
        await approveForm.validateFields();
        const reason = approveForm.getFieldValue("rejection_reason")?.trim();
        if (!reason) {
          toast.error("Rejection reason is required");
          return;
        }
      }
      const reviewed_by_id = localStorage.getItem("userId") || "SRU0002";
      const result = await approveAnnualSchedule(
        currentActionSchedule!.schedule_code,
        actionType,
        reviewed_by_id,
        actionType === "Rejected"
          ? approveForm.getFieldValue("rejection_reason")
          : undefined
      );
      toast.success(
        result?.message || `Schedule ${actionType.toLowerCase()} successfully`
      );
      setApproveModalOpen(false);
      fetchSchedules();
    } catch {
      // Error already toasted in API layer
    }
  };

  const statusTag = (status: string | null) => {
    const map: Record<string, { color: string; icon: React.ReactNode }> = {
      Draft: { color: "gold", icon: <FileTextOutlined /> },
      Approved: {
        color: "green",
        icon: <CheckCircleTwoTone twoToneColor="#52c41a" />,
      },
      Rejected: { color: "red", icon: <StopOutlined /> },
    };
    const key = status || "Draft";
    const { color, icon } = map[key] || { color: "default", icon: null };
    return (
      <Tag icon={icon} color={color}>
        {key}
      </Tag>
    );
  };

  const filteredSchedules = schedules.filter((s) => {
    const matchesSearch =
      s.year.includes(searchTerm) ||
      s.schedule_code.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = !selectedStatus || s.status === selectedStatus;
    const matchesDate =
      !dateRange ||
      (s.created_at &&
        dayjs(s.created_at).isBetween(dateRange[0], dateRange[1], "day", "[]"));
    return matchesSearch && matchesStatus && matchesDate;
  });

  const paginated = filteredSchedules.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );
  const total = schedules.length;
  const draft = schedules.filter((s) => s.status === "Draft").length;
  const approved = schedules.filter((s) => s.status === "Approved").length;
  const rejected = schedules.filter((s) => s.status === "Rejected").length;

  const columns: ColumnsType<AnnualSchedule> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) =>
            setSelectedRows(
              e.target.checked
                ? filteredSchedules.map((s) => s.schedule_code)
                : []
            )
          }
        />
      ),
      key: "selection",
      render: (_, r) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(r.schedule_code)}
          onChange={(e) =>
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, r.schedule_code]
                : prev.filter((id) => id !== r.schedule_code)
            )
          }
        />
      ),
      width: 50,
    },
    {
      title: "Year",
      dataIndex: "year",
      key: "year",
      render: (t) => <strong>{t}</strong>,
    },
    {
      title: "Code",
      dataIndex: "schedule_code",
      key: "code",
      render: (t) => <code className="bg-gray-100 px-2 py-1 rounded">{t}</code>,
    },
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      render: (_, r) => statusTag(r.status),
    },
    {
      title: "Trainings",
      key: "trainings",
      render: (_, r) => <Tag color="blue">{r.items?.length || 0}</Tag>,
    },
    {
      title: "Created",
      dataIndex: "created_at",
      key: "created",
      render: (v) => dayjs(v).format("DD-MM-YYYY"),
    },
    {
      title: "Actions",
      key: "actions",
      render: (_, r) => (
        <Space>
          <Tooltip title="Manage Items">
            <Button icon={<TeamOutlined />} onClick={() => openItemsModal(r)} />
          </Tooltip>
          {r.status === "Draft" && (
            <Tooltip title="Approve/Reject">
              <Button
                icon={<EditOutlined />}
                type="primary"
                onClick={() => openApproveModal(r)}
                className="bg-blue-600"
              />
            </Tooltip>
          )}
          <Tooltip title="Delete">
            <Button
              danger
              icon={<DeleteOutlined />}
              onClick={() => handleDeleteSchedule(r.schedule_code, r.status)}
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "year", label: "Year" },
    { key: "code", label: "Code" },
    { key: "status", label: "Status" },
    { key: "trainings", label: "Trainings" },
    { key: "created", label: "Created" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);
  const handleVisibleColumnChange = useCallback(
    (cols: string[]) => setVisibleColumnKeys(cols),
    []
  );

  const filteredColumns = columns.filter(
    (col) => col.key && visibleColumnKeys.includes(col.key as string)
  );

  const trainingItemColumns: ColumnsType<TrainingItem> = [
    {
      title: "Code - Topic",
      render: (_, r) => `${r.training_code} - ${r.training_topic}`,
      width: 320,
    },
    { title: "Month", dataIndex: "planned_month", width: 100 },
    {
      title: "Date",
      dataIndex: "planned_date",
      render: (v) => (v ? dayjs(v).format("DD-MM-YYYY") : "-"),
      width: 150,
    },
    {
      title: "Trainer",
      dataIndex: "trainer_name",
      render: (v) => v || "-",
      width: 160,
    },
    {
      title: "Status",
      dataIndex: "attendance_status",
      render: (v) => v || "Not Started",
      width: 130,
    },
    {
      title: "Remarks",
      dataIndex: "remarks",
      render: (v) => v || "-",
      width: 200,
    },
    {
      title: "Actions",
      width: 140,
      render: (_, r) => (
        <Space>
          {selectedSchedule?.status === "Draft" ? (
            <>
              <Button
                size="small"
                onClick={() => {
                  setEditingItem(r);
                  const need = trainingNeeds.find(
                    (t) => t.id === r.training_id
                  );
                  itemsForm.setFieldsValue({
                    training_id: r.training_id,
                    planned_month: r.planned_month,
                    planned_date: r.planned_date ? dayjs(r.planned_date) : null,
                    trainer_id: r.trainer_id,
                    attendance_status: r.attendance_status || "Not Started",
                    remarks: r.remarks,
                  });
                }}
              >
                Edit
              </Button>
              <Button
                size="small"
                danger
                onClick={
                  () =>
                    deleteTrainingItem(r.annual_training_item_id)
                      .then(() =>
                        reloadScheduleItems(selectedSchedule!.schedule_code)
                      )
                      .catch(() => {}) // Error already shown in API
                }
              >
                Delete
              </Button>
            </>
          ) : (
            <>
              <Button size="small" disabled>
                Edit
              </Button>
              <Button size="small" danger disabled>
                Delete
              </Button>
            </>
          )}
        </Space>
      ),
    },
  ];

  const isMobile = typeof window !== "undefined" && window.innerWidth < 1024;

  // --- CALCULATE PERCENTAGE FOR "THIS YEAR" WIDGET ---
  const currentYear = dayjs().format("YYYY");
  const currentYearSchedules = schedules.filter(
    (s) => s.year === currentYear
  ).length;
  const previousYearSchedules = schedules.filter(
    (s) => s.year === (parseInt(currentYear) - 1).toString()
  ).length;

  // Safe percentage calculation
  const thisYearPercentage =
    previousYearSchedules > 0
      ? Math.round(
          ((currentYearSchedules - previousYearSchedules) /
            previousYearSchedules) *
            100
        )
      : currentYearSchedules > 0
      ? 100
      : 0; // 100% increase if no last year data

  // Real percentage calculation
  const approvedPercentage =
    total > 0 ? Math.round((approved / total) * 100) : 0;
  const draftPercentage = total > 0 ? Math.round((draft / total) * 100) : 0;
  const rejectedPercentage =
    total > 0 ? Math.round((rejected / total) * 100) : 0;

  return (
    <>
      <div className="px-2 sm:px-4 lg:px-6 py-4">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <LuCalendarDays className="w-8 h-8 text-green-600" />
          Annual Training Schedules
          <span className="text-gray-500 text-base font-medium">({total})</span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 mb-4">
          Manage and approve annual training plans across the organization.
        </p>

        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm"
          separator=">"
          items={[
            { title: <HomeOutlined />, href: "/hrm" },
            { title: "Training" },
            { title: "Annual Schedules" },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Schedules"
            value={total}
            icon={<CalendarOutlined />}
            color="blue"
            percentage={thisYearPercentage}
            description="This year"
          />
          <WidgetCard
            title="Draft"
            value={draft}
            icon={<FileTextOutlined />}
            color="gold"
            percentage={draftPercentage}
            description="Pending approval"
          />
          <WidgetCard
            title="Approved"
            value={approved}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="green"
            percentage={approvedPercentage}
            description="Active plans"
          />
          <WidgetCard
            title="Rejected"
            value={rejected}
            icon={<WarningOutlined />}
            color="red"
            percentage={rejectedPercentage}
            description="Returned for revision"
          />
        </div>

        <Card title="Annual List" className="p-0 mb-6 shadow-sm">
          <div className="p-4 flex flex-col gap-4">
            <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
              <Input
                prefix={<SearchOutlined />}
                allowClear
                placeholder="Search by year or code"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full sm:w-64"
              />
              <Dropdown
                menu={{
                  items: [
                    {
                      key: "pdf",
                      label: (
                        <span className="flex items-center gap-2">
                          <FaFilePdf className="text-red-500" /> Export PDF
                        </span>
                      ),
                    },
                    {
                      key: "excel",
                      label: (
                        <span className="flex items-center gap-2">
                          <FaFileExcel className="text-green-500" /> Export
                          Excel
                        </span>
                      ),
                    },
                  ],
                }}
              >
                <Button icon={<FaFileExport />}>Export</Button>
              </Dropdown>
              <Button
                type="primary"
                icon={<PlusOutlined />}
                onClick={openCreateModal}
                className="bg-blue-600 hover:bg-blue-700"
              >
                New Schedule
              </Button>
            </div>

            <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
              <Select
                placeholder="Status"
                className="w-full max-w-[160px]"
                allowClear
                options={["Draft", "Approved", "Rejected"].map((s) => ({
                  label: s,
                  value: s,
                }))}
                onChange={setSelectedStatus}
              />
              <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end">
                <ColumnVisibilityToggle
                  columns={columnKeys}
                  labels={labelsMap}
                  storageKey="annual_schedule_cols"
                  onChange={handleVisibleColumnChange}
                />
                <RangePicker
                  value={dateRange}
                  onChange={(d) => setDateRange(d as any)}
                />
                <Button
                  onClick={handleBulkDelete}
                  disabled={selectedRows.length === 0}
                  danger
                  icon={<FaTrash />}
                >
                  Bulk Delete
                </Button>
                <div className="hidden sm:block">
                  <Switch
                    checkedChildren={<AppstoreOutlined />}
                    unCheckedChildren={<TableOutlined />}
                    checked={isGridView}
                    onChange={setIsGridView}
                  />
                </div>
              </div>
            </div>
          </div>

          {isGridView ? (
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
              {paginated.map((s) => (
                <GlassCard key={s.schedule_code}>
                  <div className="p-4">
                    <div className="flex justify-between items-start mb-3">
                      <h3 className="text-xl font-bold">{s.year}</h3>
                      {statusTag(s.status)}
                    </div>
                    <p>
                      <strong>Code:</strong> {s.schedule_code}
                    </p>
                    <p>
                      <strong>Trainings:</strong> {s.items?.length || 0}
                    </p>
                    <div className="mt-4 flex flex-wrap gap-2">
                      <Button
                        size="small"
                        icon={<TeamOutlined />}
                        onClick={() => openItemsModal(s)}
                      >
                        Manage
                      </Button>
                      {s.status === "Draft" && (
                        <Button
                          size="small"
                          type="primary"
                          icon={<EditOutlined />}
                          onClick={() => openApproveModal(s)}
                        >
                          Approve/Reject
                        </Button>
                      )}
                    </div>
                  </div>
                </GlassCard>
              ))}
            </div>
          ) : (
            <div className="overflow-x-auto">
              <DataTable
                columns={filteredColumns}
                dataSource={paginated}
                rowKey="schedule_code"
                scrollX="max-content"
                pagination={false}
                loading={loading}
              />
            </div>
          )}

          <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600">
            <div>
              Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
              <strong>
                {Math.min(currentPage * pageSize, filteredSchedules.length)}
              </strong>{" "}
              of <strong>{filteredSchedules.length}</strong>
            </div>
            <Pagination
              current={currentPage}
              pageSize={pageSize}
              total={filteredSchedules.length}
              onChange={(p, s) => {
                setCurrentPage(p);
                setPageSize(s);
              }}
              showSizeChanger
              pageSizeOptions={["10", "30", "50", "100"]}
              size="small"
            />
          </div>
        </Card>

        {/* Create Modal */}
        <Modal
          title="Create New Annual Training Schedule"
          open={createModalOpen}
          onOk={handleCreateSchedule}
          onCancel={() => setCreateModalOpen(false)}
          okText="Create"
          centered
          width={400}
        >
          <Form form={createForm} layout="vertical">
            <Form.Item
              name="year"
              label="Year"
              rules={[{ required: true, message: "Year is required" }]}
            >
              <Input placeholder="e.g., 2025" />
            </Form.Item>
          </Form>
        </Modal>

        {/* Approve/Reject Modal */}
        <Modal
          title={
            <span className="flex items-center gap-2">
              <EditOutlined />{" "}
              {actionType === "Approved" ? "Approve" : "Reject"} Schedule -{" "}
              {currentActionSchedule?.year}
            </span>
          }
          open={approveModalOpen}
          onOk={handleApproveSubmit}
          onCancel={() => {
            setApproveModalOpen(false);
            setCurrentActionSchedule(null);
            setActionType("Approved");
            approveForm.resetFields();
          }}
          okText={actionType === "Approved" ? "Approve" : "Reject"}
          okButtonProps={{ danger: actionType === "Rejected" }}
          centered
        >
          <div className="mb-4">
            <Button
              size="small"
              type={actionType === "Approved" ? "primary" : "default"}
              onClick={() => setActionType("Approved")}
              className="mr-2"
            >
              Approve
            </Button>
            <Button
              size="small"
              danger={actionType === "Rejected"}
              type={actionType === "Rejected" ? "primary" : "default"}
              onClick={() => setActionType("Rejected")}
            >
              Reject
            </Button>
          </div>
          <Form form={approveForm} layout="vertical">
            {actionType === "Rejected" && (
              <Form.Item
                name="rejection_reason"
                label="Rejection Reason"
                rules={[{ required: true, message: "Reason is required" }]}
              >
                <Input.TextArea
                  rows={4}
                  placeholder="Enter reason for rejection"
                />
              </Form.Item>
            )}
            {actionType === "Approved" && (
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <p className="text-blue-800 font-medium">
                  Ready to <strong>APPROVE</strong> this schedule?
                </p>
                <p className="text-sm text-blue-600 mt-2">
                  Code: <strong>{currentActionSchedule?.schedule_code}</strong>
                </p>
              </div>
            )}
          </Form>
        </Modal>

        {/* Training Items Modal */}
        <Modal
          title={`Training Items - ${selectedSchedule?.year} (${selectedSchedule?.schedule_code})`}
          open={itemsModalOpen}
          onCancel={() => {
            setItemsModalOpen(false);
            setSelectedSchedule(null);
            setEditingItem(null);
            itemsForm.resetFields();
          }}
          width={1000}
          footer={null}
          centered
          bodyStyle={{ maxHeight: "80vh", overflowY: "auto" }}
        >
          <Form form={itemsForm} layout="vertical">
            <Space wrap className="mb-4" style={{ rowGap: 16 }}>
              <Form.Item
                name="training_id"
                label="Training"
                rules={[{ required: true }]}
              >
                <Select
                  showSearch
                  allowClear
                  placeholder="Search by Employee or topic"
                  style={{ width: 360 }}
                  options={trainingNeeds.map((t) => ({
                    label: `${t.employee_name} - ${t.topic}`,
                    value: t.id,
                  }))}
                />
              </Form.Item>
              <Form.Item
                name="planned_month"
                label="Month"
                rules={[{ required: true }]}
              >
                <Select
                  placeholder="Month"
                  allowClear
                  style={{ width: 140 }}
                  options={[
                    "January",
                    "February",
                    "March",
                    "April",
                    "May",
                    "June",
                    "July",
                    "August",
                    "September",
                    "October",
                    "November",
                    "December",
                  ].map((m) => ({ label: m, value: m }))}
                />
              </Form.Item>
              <Form.Item name="planned_date" label="Date">
                <DatePicker format="DD-MM-YYYY" style={{ width: 150 }} />
              </Form.Item>
              <Form.Item name="trainer_id" label="Trainer">
                <Select
                  placeholder="Trainer"
                  allowClear
                  showSearch
                  style={{ width: 200 }}
                  options={trainers.map((t) => ({
                    label: t.name,
                    value: t.id,
                  }))}
                />
              </Form.Item>
              <Form.Item name="remarks" label="Remarks">
                <Input style={{ width: 200 }} />
              </Form.Item>
              <div className="flex items-end">
                <Button
                  type="primary"
                  onClick={editingItem ? handleUpdateItem : handleAddItem}
                  disabled={selectedSchedule?.status !== "Draft"}
                >
                  {editingItem ? "Update" : "Add"} Item
                </Button>
              </div>
            </Space>
          </Form>

          {isMobile ? (
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 mt-6">
              {(selectedSchedule?.items || []).map((item) => (
                <GlassCard key={item.annual_training_item_id}>
                  <div className="p-4">
                    <h4 className="font-semibold text-blue-700">
                      {item.training_code} - {item.training_topic}
                    </h4>
                    <p>
                      <strong>Month:</strong> {item.planned_month}
                    </p>
                    <p>
                      <strong>Date:</strong>{" "}
                      {item.planned_date
                        ? dayjs(item.planned_date).format("DD-MM-YYYY")
                        : "-"}
                    </p>
                    <p>
                      <strong>Trainer:</strong> {item.trainer_name || "-"}
                    </p>
                    {selectedSchedule?.status === "Draft" && (
                      <div className="mt-3 flex gap-2">
                        <Button
                          size="small"
                          onClick={() => {
                            setEditingItem(item);
                            itemsForm.setFieldsValue({
                              training_id: item.training_id,
                              planned_month: item.planned_month,
                              planned_date: item.planned_date
                                ? dayjs(item.planned_date)
                                : null,
                              trainer_id: item.trainer_id,
                              remarks: item.remarks,
                            });
                          }}
                        >
                          Edit
                        </Button>
                        <Button
                          size="small"
                          danger
                          onClick={
                            () =>
                              deleteTrainingItem(item.annual_training_item_id)
                                .then(() =>
                                  reloadScheduleItems(
                                    selectedSchedule!.schedule_code
                                  )
                                )
                                .catch(() => {}) // Error already shown in API
                          }
                        >
                          Delete
                        </Button>
                      </div>
                    )}
                  </div>
                </GlassCard>
              ))}
            </div>
          ) : (
            <div className="overflow-x-auto mt-4">
              <Table
                dataSource={selectedSchedule?.items || []}
                rowKey="annual_training_item_id"
                columns={trainingItemColumns}
                scroll={{ x: 1000 }}
                pagination={false}
              />
            </div>
          )}
        </Modal>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
