"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  EyeOutlined,
  SearchOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FileTextOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
  DownloadOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useRouter } from "next/navigation";
import {
  createEmployee,
  deleteEmployee,
  getAllEmployees,
  Employee,
  updateEmployee,
  downloadEmployeeFile,
} from "@/app/api/employee/employee";
import {
  getAllManageUsers,
  ManageUser,
} from "@/app/api/manageusers/manageusers";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../../components/common/GlassCard";
import { highlightMatch } from "../../components/common/HighLightMatch";
import { FaFileExcel, FaFileExport, FaFilePdf, FaTrash } from "react-icons/fa";
import { UserCheck, Users } from "lucide-react";

const WidgetCard = dynamic(() => import("../../components/common/WidgetCard"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading widget...</p>,
});
const ColumnVisibilityToggle = dynamic(
  () => import("../../components/common/ManageColumn"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
  }
);
const ConfirmDialog = dynamic(
  () => import("../../components/common/ConfirmDialog"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
  }
);

const { RangePicker } = DatePicker;
const employeeStatusOptions = ["Active", "Inactive", "Terminated"] as const;
const employeeTypeOptions = [
  "Permanent",
  "Contract",
  "Intern",
  "Trainee",
] as const;

type SafeEmployee = Employee & { employee_code: string };

export default function EmployeePage() {
  const [form] = Form.useForm();
  const [employees, setEmployees] = useState<SafeEmployee[]>([]);
  const [users, setUsers] = useState<ManageUser[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingEmployee, setEditingEmployee] = useState<SafeEmployee | null>(
    null
  );
  const [searchTerm, setSearchTerm] = useState("");
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });
  const [loading, setLoading] = useState(false);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    employeeType: undefined as string | undefined,
    employeeStatus: undefined as string | undefined,
  });
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();
  const router = useRouter();

  // Fetch data
  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        const [employeeData, userData] = await Promise.all([
          getAllEmployees(),
          getAllManageUsers(),
        ]);

        const sortedEmployees = employeeData.sort((a, b) =>
          (b.created_at || "").localeCompare(a.created_at || "")
        );

        const mappedEmployees: SafeEmployee[] = sortedEmployees
          .filter(
            (emp): emp is Employee & { employee_code: string } =>
              !!emp.employee_code
          )
          .map((emp) => ({
            ...emp,
            employee_code: emp.employee_code!,
            reporting_manager_name: emp.reporting_manager_name || null,
            created_by_name: emp.created_by_name || null,
            updated_by_name: emp.updated_by_name || null,
          }));

        setEmployees(mappedEmployees);
        setUsers(userData.filter((u) => !u.is_delete));
      } catch (error) {
        // Error already shown in API file
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  const openAddModal = () => {
    setEditingEmployee(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (employee: SafeEmployee) => {
    setEditingEmployee(employee);
    form.resetFields();
    form.setFieldsValue({
      firstName: employee.first_name || "",
      lastName: employee.last_name || "",
      designation: employee.designation || "",
      department: employee.department || "",
      email: employee.email || "",
      phone: employee.phone || "",
      dateOfBirth: employee.date_of_birth
        ? dayjs(employee.date_of_birth)
        : null,
      dateOfJoining: employee.date_of_joining
        ? dayjs(employee.date_of_joining)
        : null,
      qualification: employee.qualification || "",
      experienceYears: employee.experience_years
        ? String(employee.experience_years)
        : "",
      employeeType: employee.employee_type || "",
      employeeStatus: employee.employee_status || "",
      reportingManager: employee.reporting_manager_id || undefined,
      remarks: employee.remarks || "",
    });
    setIsModalOpen(true);
  };

  const handleDelete = (employeeCode: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Employee",
      message: "Are you sure you want to delete this employee?",
      onConfirm: async () => {
        try {
          await deleteEmployee(employeeCode);
          setEmployees((prev) =>
            prev.filter((e) => e.employee_code !== employeeCode)
          );
        } catch {
          // Error shown in API file
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Employees",
      message: `Are you sure you want to delete ${selectedRows.length} employees?`,
      onConfirm: async () => {
        try {
          await Promise.all(selectedRows.map(deleteEmployee));
          setEmployees((prev) =>
            prev.filter((e) => !selectedRows.includes(e.employee_code))
          );
          setSelectedRows([]);
        } catch {
          // Error from API
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();

      const hasFiles =
        values.resumePath?.file?.originFileObj ||
        values.qualificationDocPath?.file?.originFileObj ||
        values.idProofPath?.file?.originFileObj;

      let payload: any;

      if (hasFiles) {
        payload = new FormData();
        const fields = {
          first_name: values.firstName,
          last_name: values.lastName || "",
          designation: values.designation || "",
          department: values.department || "",
          email: values.email,
          phone: values.phone,
          date_of_birth: values.dateOfBirth
            ? values.dateOfBirth.format("YYYY-MM-DD")
            : "",
          date_of_joining: values.dateOfJoining
            ? values.dateOfJoining.format("YYYY-MM-DD")
            : "",
          qualification: values.qualification || "",
          experience_years: values.experienceYears || "",
          employee_type: values.employeeType || "",
          employee_status: values.employeeStatus || "",
          reporting_manager: values.reportingManager || "",
          remarks: values.remarks || "",
        };

        Object.entries(fields).forEach(([key, value]) => {
          if (value !== undefined && value !== null && value !== "") {
            payload.append(key, value);
          }
        });

        if (values.resumePath?.file?.originFileObj)
          payload.append("resume", values.resumePath.file.originFileObj);
        if (values.qualificationDocPath?.file?.originFileObj)
          payload.append(
            "qualification_docx",
            values.qualificationDocPath.file.originFileObj
          );
        if (values.idProofPath?.file?.originFileObj)
          payload.append("id_proof", values.idProofPath.file.originFileObj);
      } else {
        payload = {
          first_name: values.firstName || null,
          last_name: values.lastName || null,
          designation: values.designation || null,
          department: values.department || null,
          email: values.email || null,
          phone: values.phone || null,
          date_of_birth: values.dateOfBirth
            ? values.dateOfBirth.format("YYYY-MM-DD")
            : null,
          date_of_joining: values.dateOfJoining
            ? values.dateOfJoining.format("YYYY-MM-DD")
            : null,
          qualification: values.qualification || null,
          experience_years: values.experienceYears
            ? Number(values.experienceYears)
            : null,
          employee_type: values.employeeType || null,
          employee_status: values.employeeStatus || null,
          reporting_manager: values.reportingManager || null,
          remarks: values.remarks || null,
        };
      }

      if (editingEmployee) {
        await updateEmployee(editingEmployee.employee_code, payload);
      } else {
        await createEmployee(payload);
      }

      const updatedEmployees = await getAllEmployees();
      const sorted = updatedEmployees.sort((a, b) =>
        (b.created_at || "").localeCompare(a.created_at || "")
      );
      const safeEmployees: SafeEmployee[] = sorted
        .filter(
          (e): e is Employee & { employee_code: string } => !!e.employee_code
        )
        .map((e) => ({ ...e, employee_code: e.employee_code! }));

      setEmployees(safeEmployees);
      setIsModalOpen(false);
      form.resetFields();
      setEditingEmployee(null);
    } catch (error: any) {
      if (error.errorFields?.length > 0) {
        const missing = error.errorFields.map((f: any) => {
          const name = f.name[0];
          const labels: Record<string, string> = {
            firstName: "First Name",
            lastName: "Last Name",
            email: "Email",
            phone: "Phone",
            employeeType: "Employee Type",
            employeeStatus: "Employee Status",
          };
          return labels[name] || name;
        });
        toast.error(
          missing.length > 1
            ? `${missing.join(", ")} are required`
            : `${missing[0]} is required`
        );
      }
    }
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Active: "green",
      Inactive: "gold",
      Terminated: "red",
      Permanent: "blue",
      Contract: "cyan",
      Intern: "purple",
      Trainee: "orange",
    };
    const iconMap: Record<string, React.ReactNode> = {
      Active: <CheckCircleOutlined />,
      Inactive: <MinusCircleOutlined />,
      Terminated: <CloseCircleOutlined />,
      Permanent: <CheckCircleOutlined />,
      Contract: <FileTextOutlined />,
      Intern: <PauseCircleOutlined />,
      Trainee: <FileTextOutlined />,
    };
    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  // Filter employees
  const filteredEmployees = employees.filter((employee) => {
    const fullName = `${employee.first_name || ""} ${
      employee.last_name || ""
    }`.trim();
    const matchesSearch =
      fullName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      (employee.first_name || "")
        .toLowerCase()
        .includes(searchTerm.toLowerCase()) ||
      (employee.last_name || "")
        .toLowerCase()
        .includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (employee.created_at &&
        (!dateRange[0] ||
          dayjs(employee.created_at).isAfter(dateRange[0], "day")) &&
        (!dateRange[1] ||
          dayjs(employee.created_at).isBefore(dateRange[1], "day")));
    const matchesAdvanced =
      (!advancedFilters.employeeType ||
        employee.employee_type === advancedFilters.employeeType) &&
      (!advancedFilters.employeeStatus ||
        employee.employee_status === advancedFilters.employeeStatus);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedEmployees = filteredEmployees.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  // Columns
  const columns: ColumnsType<SafeEmployee> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) =>
            setSelectedRows(
              e.target.checked
                ? filteredEmployees.map((e) => e.employee_code)
                : []
            )
          }
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.employee_code)}
          onChange={(e) =>
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.employee_code]
                : prev.filter((id) => id !== record.employee_code)
            )
          }
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Name",
      key: "name",
      render: (_, record) => {
        const fullName = `${record.first_name || ""} ${
          record.last_name || ""
        }`.trim();
        return (
          <Button
            type="link"
            className="text-green-600 font-medium p-0"
            onClick={() =>
              router.push(
                `/hrm/employeemaster/employeedetail?id=${
                  record.employee_code
                }&name=${encodeURIComponent(fullName)}`
              )
            }
          >
            {highlightMatch(fullName, searchTerm)}
          </Button>
        );
      },
    },
    { title: "Designation", dataIndex: "designation", key: "designation" },
    { title: "Department", dataIndex: "department", key: "department" },
    {
      title: "Employee Type",
      dataIndex: "employee_type",
      key: "employee_type",
      render: renderTag,
    },
    { title: "Email", dataIndex: "email", key: "email" },
    { title: "Phone", dataIndex: "phone", key: "phone" },
    {
      title: "Employee Status",
      dataIndex: "employee_status",
      key: "employee_status",
      render: renderTag,
    },
    {
      title: "Reporting Manager",
      dataIndex: "reporting_manager_name",
      key: "reporting_manager_name",
      render: (t) => t || "-",
    },
    {
      title: "Date of Birth",
      dataIndex: "date_of_birth",
      key: "date_of_birth",
      render: (v) => (v ? dayjs(v).format("DD-MM-YYYY") : "-"),
    },
    {
      title: "Qualification",
      dataIndex: "qualification",
      key: "qualification",
      render: (t) => t || "-",
    },
    {
      title: "Date of Joining",
      dataIndex: "date_of_joining",
      key: "date_of_joining",
      render: (v) => (v ? dayjs(v).format("DD-MM-YYYY") : "-"),
    },
    {
      title: "Experience (Years)",
      dataIndex: "experience_years",
      key: "experience_years",
      render: (v) => v ?? "-",
    },
    {
      title: "Resume",
      dataIndex: "resume_path",
      key: "resume_path",
      render: (v, r) =>
        v ? (
          <Button
            type="link"
            icon={<DownloadOutlined />}
            onClick={() => downloadEmployeeFile(r.employee_code, "resume")}
          >
            Download
          </Button>
        ) : (
          <Tag color="red">Pending</Tag>
        ),
    },
    {
      title: "Qualification Docx",
      dataIndex: "qualification_doc_path",
      key: "qualification_doc_path",
      render: (v, r) =>
        v ? (
          <Button
            type="link"
            icon={<DownloadOutlined />}
            onClick={() =>
              downloadEmployeeFile(r.employee_code, "qualification")
            }
          >
            Download
          </Button>
        ) : (
          <Tag color="red">Pending</Tag>
        ),
    },
    {
      title: "ID Proof",
      dataIndex: "id_proof_path",
      key: "id_proof_path",
      render: (v, r) =>
        v ? (
          <Button
            type="link"
            icon={<DownloadOutlined />}
            onClick={() => downloadEmployeeFile(r.employee_code, "id_proof")}
          >
            Download
          </Button>
        ) : (
          <Tag color="red">Pending</Tag>
        ),
    },
    {
      title: "Remarks",
      dataIndex: "remarks",
      key: "remarks",
      render: (t) => t || "-",
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      render: (v) =>
        v
          ? `${dayjs(v).format("DD-MM-YYYY hh:mm A")} (${dayjs(v).fromNow()})`
          : "-",
    },
    {
      title: "Created By",
      dataIndex: "created_by_name",
      key: "created_by_name",
      render: (t) => t || "-",
    },
    {
      title: "Updated By",
      dataIndex: "updated_by_name",
      key: "updated_by_name",
      render: (t) => t || "-",
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => {
        const fullName = `${record.first_name || ""} ${
          record.last_name || ""
        }`.trim();
        return (
          <Space size="small">
            <Tooltip title="View">
              <Button
                icon={<EyeOutlined />}
                onClick={() =>
                  router.push(
                    `/hrm/employeemaster/employeedetail?id=${
                      record.employee_code
                    }&name=${encodeURIComponent(fullName)}`
                  )
                }
                size="small"
                className="text-blue-600"
              />
            </Tooltip>
            <Tooltip title="Edit">
              <Button
                icon={<EditOutlined />}
                onClick={() => openEditModal(record)}
                size="small"
                type="primary"
                className="bg-blue-600 hover:bg-blue-700"
              />
            </Tooltip>
            <Tooltip title="Delete">
              <Button
                icon={<DeleteOutlined />}
                onClick={() => handleDelete(record.employee_code)}
                danger
                size="small"
                className="text-red-600"
              />
            </Tooltip>
          </Space>
        );
      },
    },
  ];

  // Column visibility
  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "name", label: "Name" },
    { key: "designation", label: "Designation" },
    { key: "department", label: "Department" },
    { key: "employee_type", label: "Employee Type" },
    { key: "email", label: "Email" },
    { key: "phone", label: "Phone" },
    { key: "employee_status", label: "Employee Status" },
    { key: "reporting_manager_name", label: "Reporting Manager" },
    { key: "date_of_birth", label: "Date of Birth" },
    { key: "qualification", label: "Qualification" },
    { key: "date_of_joining", label: "Date of Joining" },
    { key: "experience_years", label: "Experience (Years)" },
    { key: "resume_path", label: "Resume" },
    { key: "qualification_doc_path", label: "Qualification Docx" },
    { key: "id_proof_path", label: "ID Proof" },
    { key: "remarks", label: "Remarks" },
    { key: "created_at", label: "Created At" },
    { key: "created_by_name", label: "Created By" },
    { key: "updated_by_name", label: "Updated By" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);
  const handleVisibleColumnChange = useCallback(
    (cols: string[]) => setVisibleColumnKeys(cols),
    []
  );

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  // Grid Card
  const EmployeeCard = ({ employee }: { employee: SafeEmployee }) => {
    const fullName = `${employee.first_name || ""} ${
      employee.last_name || ""
    }`.trim();
    return (
      <GlassCard gradient blurStrength="lg" hoverLift color="success">
        <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full space-y-2">
          <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2">
            <h3 className="text-lg font-semibold text-gray-800 dark:text-white">
              {fullName}
            </h3>
            <Tag color="blue">{employee.department || "N/A"}</Tag>
          </div>
          <p className="text-sm text-gray-600 dark:text-gray-300">
            <strong>Type:</strong> {renderTag(employee.employee_type || "N/A")}
          </p>
          <p className="text-sm text-gray-600 dark:text-gray-300">
            <strong>Status:</strong>{" "}
            {renderTag(employee.employee_status || "N/A")}
          </p>
          <p className="text-sm text-gray-600 dark:text-gray-300">
            <strong>Designation:</strong> {employee.designation || "N/A"}
          </p>
          <p className="text-sm text-gray-600 dark:text-gray-300">
            <strong>Email:</strong> {employee.email || "N/A"}
          </p>
          <p className="text-sm text-gray-600 dark:text-gray-300">
            <strong>Phone:</strong> {employee.phone || "N/A"}
          </p>
          <p className="text-sm text-gray-600 dark:text-gray-300">
            <strong>Reporting Manager:</strong>{" "}
            {employee.reporting_manager_name || "N/A"}
          </p>
          <div className="flex gap-2 pt-2 border-t dark:border-gray-700">
            <Tooltip title="View">
              <Button
                icon={<EyeOutlined />}
                onClick={() =>
                  router.push(
                    `/hrm/employeemaster/employeedetail?id=${
                      employee.employee_code
                    }&name=${encodeURIComponent(fullName)}`
                  )
                }
                size="small"
                className="text-blue-600"
              />
            </Tooltip>
            <Tooltip title="Edit">
              <Button
                icon={<EditOutlined />}
                onClick={() => openEditModal(employee)}
                size="small"
                type="primary"
                className="bg-blue-600 hover:bg-blue-700"
              />
            </Tooltip>
            <Tooltip title="Delete">
              <Button
                icon={<DeleteOutlined />}
                onClick={() => handleDelete(employee.employee_code)}
                danger
                size="small"
                className="text-red-600"
              />
            </Tooltip>
          </div>
        </div>
      </GlassCard>
    );
  };

  const total = employees.length;
  const active = employees.filter((e) => e.employee_status === "Active").length;
  const terminated = employees.filter(
    (e) => e.employee_status === "Terminated"
  ).length;
  const permanent = employees.filter(
    (e) => e.employee_type === "Permanent"
  ).length;

  const downloadFile = (data: Blob, fileName: string) => {
    const url = window.URL.createObjectURL(data);
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", fileName);
    document.body.appendChild(link);
    link.click();
    link.remove();
  };

  function handleExport(key: string): void {
    throw new Error("Function not implemented.");
  }

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <Users className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Employees</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({employees.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage employee master data used throughout the system.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/hrm",
            },
            { title: "Master Data" },
            { title: "Employees" },
          ]}
        />
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Employees"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All registered employees"
          />
          <WidgetCard
            title="Active"
            value={active}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Currently active employees"
          />
          <WidgetCard
            title="Terminated"
            value={terminated}
            icon={<StopOutlined />}
            color="red"
            percentage={-3}
            description="Employees no longer employed"
          />
          <WidgetCard
            title="Permanent"
            value={permanent}
            icon={<UserCheck />}
            color="orange"
            percentage={5}
            description="Permanent employees"
          />
        </div>
        <div className="mt-4">
          <Card title="Employee List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Employee by name"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  menu={{
                    items: [
                      {
                        key: "pdf",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFilePdf className="text-red-500" /> Export PDF
                          </span>
                        ),
                      },
                      {
                        key: "excel",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFileExcel className="text-green-500" /> Export
                            Excel
                          </span>
                        ),
                      },
                    ],
                    onClick: ({ key }) => handleExport(key),
                  }}
                >
                  <Button type="default" icon={<FaFileExport />}>
                    Export
                  </Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Employee
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Employee Type"
                    className="w-full max-w-[160px]"
                    onChange={(v) =>
                      setAdvancedFilters((p) => ({ ...p, employeeType: v }))
                    }
                    allowClear
                    options={employeeTypeOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                  <Select
                    placeholder="Employee Status"
                    className="w-full max-w-[160px]"
                    onChange={(v) =>
                      setAdvancedFilters((p) => ({ ...p, employeeStatus: v }))
                    }
                    allowClear
                    options={employeeStatusOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="employee_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(d) => setDateRange(d as any)}
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    icon={<FaTrash />}
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>
            </div>
            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedEmployees.map((employee) => (
                  <EmployeeCard
                    key={employee.employee_code}
                    employee={employee}
                  />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<Employee>
                  columns={filteredColumns}
                  dataSource={paginatedEmployees}
                  rowKey="employee_code"
                  scrollX="max-content"
                  className="mt-2"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(currentPage * pageSize, filteredEmployees.length)}
                </strong>{" "}
                of <strong>{filteredEmployees.length}</strong> employees
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredEmployees.length}
                onChange={(p, s) => {
                  setCurrentPage(p);
                  setPageSize(s);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <Users
                  className={`w-5 h-5 ${
                    !editingEmployee ? "text-green-600" : "text-gray-400"
                  }`}
                />
                <UserCheck
                  className={`w-5 h-5 ${
                    editingEmployee ? "text-blue-600" : "text-gray-400"
                  }`}
                />
                <span>
                  {editingEmployee ? "Edit Employee" : "Add Employee"}
                </span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            destroyOnHidden
            width="95vw"
            className="!top-5 font-semibold"
            styles={{ body: { maxHeight: "80vh", overflowY: "auto" } }}
          >
            <Form form={form} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Personal Details
                  </h3>
                </div>
                <Form.Item
                  name="firstName"
                  label="First Name"
                  rules={[
                    { required: true, message: "First Name is required" },
                  ]}
                >
                  <Input />
                </Form.Item>
                <Form.Item name="lastName" label="Last Name">
                  <Input />
                </Form.Item>
                <Form.Item
                  name="email"
                  label="Email"
                  rules={[
                    { required: true, message: "Email is required" },
                    { type: "email", message: "Invalid email format" },
                  ]}
                >
                  <Input type="email" />
                </Form.Item>
                <Form.Item
                  name="phone"
                  label="Phone"
                  rules={[
                    { required: true, message: "Phone number is required" },
                    {
                      pattern: /^[0-9]{10,15}$/,
                      message: "Phone number must be 10 to 15 digits",
                    },
                  ]}
                >
                  <Input />
                </Form.Item>
                <Form.Item name="dateOfBirth" label="Date of Birth">
                  <DatePicker format="YYYY-MM-DD" className="w-full" />
                </Form.Item>
                <Form.Item name="qualification" label="Qualification">
                  <Input />
                </Form.Item>

                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Professional Details
                  </h3>
                </div>
                <Form.Item name="designation" label="Designation">
                  <Input />
                </Form.Item>
                <Form.Item name="department" label="Department">
                  <Input />
                </Form.Item>
                <Form.Item name="dateOfJoining" label="Date of Joining">
                  <DatePicker format="YYYY-MM-DD" className="w-full" />
                </Form.Item>
                <Form.Item name="experienceYears" label="Experience (Years)">
                  <Input type="number" min={0} />
                </Form.Item>
                <Form.Item
                  name="employeeType"
                  label="Employee Type"
                  rules={[
                    { required: true, message: "Employee Type is required" },
                  ]}
                >
                  <Select
                    options={employeeTypeOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </Form.Item>
                <Form.Item
                  name="employeeStatus"
                  label="Employee Status"
                  rules={[{ required: true, message: "Status is required" }]}
                >
                  <Select
                    options={employeeStatusOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </Form.Item>
                <Form.Item name="reportingManager" label="Reporting Manager">
                  <Select
                    showSearch
                    placeholder="Select a reporting manager"
                    optionFilterProp="children"
                    filterOption={(input, option) =>
                      (option?.label ?? "")
                        .toLowerCase()
                        .includes(input.toLowerCase())
                    }
                    options={users
                      .filter(
                        (user) =>
                          !editingEmployee ||
                          user.id !== editingEmployee.employee_code
                      )
                      .map((user) => ({
                        label: `${user.First_name} ${user.Last_name || ""}`,
                        value: user.id,
                      }))}
                    allowClear
                  />
                </Form.Item>

                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">Documents</h3>
                </div>
                <Form.Item
                  name="resumePath"
                  label="Resume"
                  getValueFromEvent={(e) =>
                    e?.fileList?.[0]?.originFileObj || null
                  }
                >
                  {editingEmployee?.resume_path && (
                    <div className="mb-2">
                      <Button
                        type="link"
                        icon={<DownloadOutlined />}
                        onClick={() =>
                          editingEmployee.employee_code &&
                          downloadEmployeeFile(
                            editingEmployee.employee_code,
                            "resume"
                          )
                        }
                      >
                        Download Existing File
                      </Button>
                    </div>
                  )}
                  <Input type="file" accept=".pdf,.doc,.docx" />
                </Form.Item>
                <Form.Item
                  name="qualificationDocPath"
                  label="Qualification Docx"
                  getValueFromEvent={(e) =>
                    e?.fileList?.[0]?.originFileObj || null
                  }
                >
                  {editingEmployee?.qualification_doc_path && (
                    <div className="mb-2">
                      <Button
                        type="link"
                        icon={<DownloadOutlined />}
                        onClick={() =>
                          editingEmployee.employee_code &&
                          downloadEmployeeFile(
                            editingEmployee.employee_code,
                            "qualification"
                          )
                        }
                      >
                        Download Existing File
                      </Button>
                    </div>
                  )}
                  <Input type="file" accept=".pdf,.doc,.docx" />
                </Form.Item>
                <Form.Item
                  name="idProofPath"
                  label="ID Proof"
                  getValueFromEvent={(e) =>
                    e?.fileList?.[0]?.originFileObj || null
                  }
                >
                  {editingEmployee?.id_proof_path && (
                    <div className="mb-2">
                      <Button
                        type="link"
                        icon={<DownloadOutlined />}
                        onClick={() =>
                          editingEmployee.employee_code &&
                          downloadEmployeeFile(
                            editingEmployee.employee_code,
                            "id_proof"
                          )
                        }
                      >
                        Download Existing File
                      </Button>
                    </div>
                  )}
                  <Input type="file" accept=".pdf,.jpg,.jpeg,.png" />
                </Form.Item>
                <div className="col-span-full">
                  <Form.Item name="remarks" label="Remarks">
                    <Input.TextArea rows={2} />
                  </Form.Item>
                </div>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
