"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  WarningOutlined,
  EyeOutlined,
  SearchOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FileTextOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useRouter } from "next/navigation";
import {
  createEmployeeSkill,
  deleteEmployeeSkill,
  getAllEmployeeSkills,
  EmployeeSkill,
  updateEmployeeSkill,
} from "@/app/api/employeeskillmatrix/employeeskillmatrix";
import { getAllEmployees, Employee } from "@/app/api/employee/employee";
import {
  getAllManageUsers,
  ManageUser,
} from "@/app/api/manageusers/manageusers";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../../components/common/GlassCard";
import { highlightMatch } from "../../components/common/HighLightMatch";
import { FaFileExcel, FaFileExport, FaFilePdf, FaTrash } from "react-icons/fa";
import { UserCheck, Users } from "lucide-react";

const WidgetCard = dynamic(() => import("../../components/common/WidgetCard"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading widget...</p>,
});

const ColumnVisibilityToggle = dynamic(
  () => import("../../components/common/ManageColumn"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
  }
);

const ConfirmDialog = dynamic(
  () => import("../../components/common/ConfirmDialog"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
  }
);

const { RangePicker } = DatePicker;
const proficiencyLevelOptions = ["Basic", "Intermediate", "Expert"] as const;

export default function EmployeeSkillMatrixPage() {
  const [form] = Form.useForm();
  const [employeeSkills, setEmployeeSkills] = useState<EmployeeSkill[]>([]);
  const [users, setUsers] = useState<Employee[]>([]);
  const [verifiers, setVerifiers] = useState<ManageUser[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingSkill, setEditingSkill] = useState<EmployeeSkill | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });
  const [loading, setLoading] = useState(false);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    skillCategory: undefined as string | undefined,
    proficiencyLevel: undefined as string | undefined,
  });
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();
  const router = useRouter();

  const fetchData = async () => {
    try {
      setLoading(true);
      const [skillData, userData, verifierData] = await Promise.all([
        getAllEmployeeSkills(),
        getAllEmployees(),
        getAllManageUsers(),
      ]);

      const sortedSkills = skillData.sort(
        (a: any, b: any) =>
          new Date(b.created_at || "").getTime() -
          new Date(a.created_at || "").getTime()
      );

      setEmployeeSkills(sortedSkills);
      setUsers(userData);
      setVerifiers(verifierData.filter((u) => u.is_delete !== true));
    } catch (error) {
      // Backend errors already handled in API file
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
  }, []);

  const openAddModal = () => {
    setEditingSkill(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (skill: EmployeeSkill) => {
    setEditingSkill(skill);
    form.resetFields();
    form.setFieldsValue({
      skillName: skill.skill_name || "",
      skillCategory: skill.skill_category || "",
      proficiencyLevel: skill.proficiency_level || "",
      lastAssessedDate: skill.last_assessed_date
        ? dayjs(skill.last_assessed_date)
        : null,
      remarks: skill.remarks || "",
    });
    setIsModalOpen(true);
  };

  const handleDelete = (skillCode: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Employee Skill",
      message: "Are you sure you want to delete this employee skill?",
      onConfirm: async () => {
        try {
          await deleteEmployeeSkill(skillCode);
          await fetchData();
          toast.success("Employee skill deleted successfully");
        } catch {
          // Handled in API
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Employee Skills",
      message: `Are you sure you want to delete ${selectedRows.length} employee skills?`,
      onConfirm: async () => {
        try {
          await Promise.all(
            selectedRows.map((code) => deleteEmployeeSkill(code))
          );
          await fetchData();
          setSelectedRows([]);
          toast.success("Employee skills deleted successfully");
        } catch {
          // Handled in API
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();

      const payload: Partial<EmployeeSkill> = {
        employee_code: editingSkill ? undefined : values.employeeCode,
        skill_name: values.skillName,
        skill_category: values.skillCategory,
        proficiency_level: values.proficiencyLevel,
        last_assessed_date: values.lastAssessedDate
          ? values.lastAssessedDate.format("YYYY-MM-DD")
          : null,
        verified_by: editingSkill
          ? undefined
          : values.verifiedBy
          ? Number(values.verifiedBy)
          : null,
        remarks: values.remarks || null,
      };

      if (editingSkill && editingSkill.skill_code) {
        const cleanPayload = Object.fromEntries(
          Object.entries(payload).filter(([_, v]) => v !== undefined)
        );
        await updateEmployeeSkill(editingSkill.skill_code, cleanPayload);
        toast.success("Employee skill updated successfully");
      } else {
        await createEmployeeSkill(payload as EmployeeSkill);
        toast.success("Employee skill added successfully");
      }

      await fetchData();
      setIsModalOpen(false);
      form.resetFields();
      setEditingSkill(null);
    } catch (err: any) {
      if (err.errorFields && err.errorFields.length > 0) {
        const missingFields = err.errorFields.map((field: any) => {
          const fieldName = field.name[0];
          const labelMap: Record<string, string> = {
            employeeCode: "Employee",
            skillName: "Skill Name",
            skillCategory: "Skill Category",
            proficiencyLevel: "Proficiency Level",
          };
          return labelMap[fieldName] || fieldName;
        });

        const message =
          missingFields.length > 1
            ? `${missingFields
                .slice(0, -1)
                .join(", ")} and ${missingFields.slice(-1)} are required`
            : `${missingFields[0]} is required`;

        toast.error(message);
      }
    }
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Basic: "red",
      Intermediate: "gold",
      Expert: "purple",
    };
    const iconMap: Record<string, React.ReactNode> = {
      Basic: <MinusCircleOutlined />,
      Intermediate: <PauseCircleOutlined />,
      Expert: <CheckCircleTwoTone twoToneColor="#800080" />,
    };
    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredSkills = employeeSkills.filter((skill) => {
    const matchesSearch =
      (skill.employee_name || "")
        .toLowerCase()
        .includes(searchTerm.toLowerCase()) ||
      (skill.skill_name || "").toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (skill.created_at &&
        (!dateRange[0] ||
          dayjs(skill.created_at).isSame(dateRange[0], "day") ||
          dayjs(skill.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(skill.created_at).isSame(dateRange[1], "day") ||
          dayjs(skill.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.skillCategory ||
        skill.skill_category === advancedFilters.skillCategory) &&
      (!advancedFilters.proficiencyLevel ||
        skill.proficiency_level === advancedFilters.proficiencyLevel);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedSkills = filteredSkills.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<EmployeeSkill> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(
              checked ? filteredSkills.map((s) => s.skill_code!) : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.skill_code!)}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.skill_code!]
                : prev.filter((id) => id !== record.skill_code!)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Employee Name",
      key: "employee_name",
      sorter: (a, b) =>
        (a.employee_name || "").localeCompare(b.employee_name || ""),
      render: (_, record) => (
        <Button
          type="link"
          className="text-green-600 font-medium p-0"
          onClick={() =>
            router.push(
              `/hrm/employeemaster/employeedetail?id=${
                record.employee_code
              }&name=${encodeURIComponent(record.employee_name || "")}`
            )
          }
        >
          {highlightMatch(record.employee_name || "", searchTerm)}
        </Button>
      ),
      filterSearch: true,
      onFilter: (value, record) =>
        (record.employee_name || "")
          .toLowerCase()
          .includes((value as string).toLowerCase()),
      filters: Array.from(
        new Set(employeeSkills.map((s) => s.employee_name || ""))
      ).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Skill Name",
      dataIndex: "skill_name",
      key: "skill_name",
      sorter: (a, b) => (a.skill_name || "").localeCompare(b.skill_name || ""),
      render: (text) => highlightMatch(text || "", searchTerm),
    },
    {
      title: "Skill Category",
      dataIndex: "skill_category",
      key: "skill_category",
      sorter: (a, b) =>
        (a.skill_category || "").localeCompare(b.skill_category || ""),
      render: (text) => highlightMatch(text || "", searchTerm),
    },
    {
      title: "Proficiency Level",
      dataIndex: "proficiency_level",
      key: "proficiency_level",
      sorter: (a, b) =>
        (a.proficiency_level || "").localeCompare(b.proficiency_level || ""),
      render: renderTag,
      filters: proficiencyLevelOptions.map((t) => ({ text: t, value: t })),
      onFilter: (value, record) => record.proficiency_level === value,
    },
    {
      title: "Last Assessed",
      dataIndex: "last_assessed_date",
      key: "last_assessed_date",
      sorter: (a, b) =>
        dayjs(a.last_assessed_date || "").unix() -
        dayjs(b.last_assessed_date || "").unix(),
      render: (value: string) =>
        value ? dayjs(value).format("DD-MM-YYYY") : "-",
    },
    {
      title: "Verified By",
      dataIndex: "verified_by_name",
      key: "verified_by_name",
      sorter: (a, b) =>
        (a.verified_by_name || "").localeCompare(b.verified_by_name || ""),
      render: (text) => text || "-",
    },
    {
      title: "Remarks",
      dataIndex: "remarks",
      key: "remarks",
      sorter: (a, b) => (a.remarks || "").localeCompare(b.remarks || ""),
      render: (text) => text || "-",
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      sorter: (a, b) => dayjs(a.created_at).unix() - dayjs(b.created_at).unix(),
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.skill_code!)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "employee_name", label: "Employee Name" },
    { key: "skill_name", label: "Skill Name" },
    { key: "skill_category", label: "Skill Category" },
    { key: "proficiency_level", label: "Proficiency Level" },
    { key: "last_assessed_date", label: "Last Assessed" },
    { key: "verified_by_name", label: "Verified By" },
    { key: "remarks", label: "Remarks" },
    { key: "created_at", label: "Created At" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const SkillCard = ({ skill }: { skill: EmployeeSkill }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="success">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full space-y-2">
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2">
          <h3 className="text-lg font-semibold text-gray-800 dark:text-white">
            {skill.employee_name || "N/A"}
          </h3>
          <Tag color="blue">{skill.skill_name || "N/A"}</Tag>
        </div>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Skill Category:</strong> {skill.skill_category || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Proficiency Level:</strong>{" "}
          {renderTag(skill.proficiency_level || "N/A")}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Last Assessed:</strong>{" "}
          {skill.last_assessed_date
            ? dayjs(skill.last_assessed_date).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Verified By:</strong> {skill.verified_by_name || "N/A"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Remarks:</strong> {skill.remarks || "N/A"}
        </p>
        <p className="text-xs text-gray-500 dark:text-gray-400 mt-2">
          <strong>Created:</strong>{" "}
          {skill.created_at
            ? dayjs(skill.created_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <div className="flex gap-2 pt-2 border-t dark:border-gray-700">
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(skill)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(skill.skill_code!)}
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
        </div>
      </div>
    </GlassCard>
  );

  const total = employeeSkills.length;
  const expert = employeeSkills.filter(
    (s) => s.proficiency_level === "Expert"
  ).length;
  const intermediate = employeeSkills.filter(
    (s) => s.proficiency_level === "Intermediate"
  ).length;
  const beginner = employeeSkills.filter(
    (s) => s.proficiency_level === "Basic"
  ).length;

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <Users className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Employee Skill Matrix</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({employeeSkills.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage employee skill matrix data used throughout the system.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/hrm",
            },
            {
              title: "Master Data",
            },
            {
              title: "Employee Skill Matrix",
            },
          ]}
        />
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Skills"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={10}
            description="All recorded employee skills"
          />
          <WidgetCard
            title="Expert Level"
            value={expert}
            icon={<FileTextOutlined />}
            color="blue"
            percentage={5}
            description="Skills in expert category"
          />
          <WidgetCard
            title="Intermediate Level"
            value={intermediate}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="orange"
            percentage={3}
            description="Employees with intermediate proficiency"
          />
          <WidgetCard
            title="Basic Level"
            value={beginner}
            icon={<UserCheck />}
            color="purple"
            percentage={7}
            description="Skills in basic category"
          />
        </div>

        <div className="mt-4">
          <Card title="Employee Skill List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search by employee or skill name"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      {
                        key: "pdf",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFilePdf className="text-red-500" /> Export PDF
                          </span>
                        ),
                      },
                      {
                        key: "excel",
                        label: (
                          <span className="flex items-center gap-2">
                            <FaFileExcel className="text-green-500" /> Export
                            Excel
                          </span>
                        ),
                      },
                    ],
                  }}
                >
                  <Button type="default" icon={<FaFileExport />}>
                    Export
                  </Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Skill
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Proficiency Level"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        proficiencyLevel: value,
                      }))
                    }
                    allowClear
                    options={proficiencyLevelOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="skill_matrix_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    icon={<FaTrash />}
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedSkills.map((skill) => (
                  <SkillCard key={skill.skill_code} skill={skill} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<EmployeeSkill>
                  columns={filteredColumns}
                  dataSource={paginatedSkills}
                  rowKey="skill_code"
                  scrollX="max-content"
                  className="mt-2"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(currentPage * pageSize, filteredSkills.length)}
                </strong>{" "}
                of <strong>{filteredSkills.length}</strong> skills
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredSkills.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <div className="flex items-center gap-1">
                  <Users
                    className={`w-5 h-5 ${
                      !editingSkill ? "text-green-600" : "text-gray-400"
                    }`}
                  />
                  <UserCheck
                    className={`w-5 h-5 ${
                      editingSkill ? "text-blue-600" : "text-gray-400"
                    }`}
                  />
                </div>
                <span>
                  {editingSkill ? "Edit Employee Skill" : "Add Employee Skill"}
                </span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
              setEditingSkill(null);
            }}
            destroyOnHidden
            width="95vw"
            className="!top-5 font-semibold"
            styles={{ body: { maxHeight: "80vh", overflowY: "auto" } }}
          >
            <Form form={form} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">Skill Details</h3>
                </div>

                {/* EDIT MODE: Show Skill Code (readonly) */}
                {editingSkill && (
                  <Form.Item label="Skill Code">
                    <Input value={editingSkill.skill_code} disabled />
                  </Form.Item>
                )}

                {/* ADD MODE: Employee + Verified By */}
                {!editingSkill && (
                  <>
                    <Form.Item
                      name="employeeCode"
                      label="Employee"
                      rules={[
                        { required: true, message: "Employee is required" },
                      ]}
                    >
                      <Select
                        showSearch
                        placeholder="Select an employee"
                        optionFilterProp="children"
                        filterOption={(input, option) =>
                          (option?.label ?? "")
                            .toLowerCase()
                            .includes(input.toLowerCase())
                        }
                        options={users.map((user) => ({
                          label: `${user.first_name} ${user.last_name || ""}`,
                          value: user.employee_code,
                        }))}
                      />
                    </Form.Item>

                    <Form.Item name="verifiedBy" label="Verified By">
                      <Select
                        showSearch
                        placeholder="Select a verifier"
                        optionFilterProp="children"
                        filterOption={(input, option) =>
                          (option?.label ?? "")
                            .toLowerCase()
                            .includes(input.toLowerCase())
                        }
                        options={verifiers.map((verifier) => ({
                          label: `${verifier.First_name} ${
                            verifier.Last_name || ""
                          }`,
                          value: verifier.id,
                        }))}
                        allowClear
                      />
                    </Form.Item>
                  </>
                )}

                <Form.Item
                  name="skillName"
                  label="Skill Name"
                  rules={[
                    { required: true, message: "Skill Name is required" },
                  ]}
                >
                  <Input />
                </Form.Item>

                <Form.Item name="skillCategory" label="Skill Category">
                  <Input placeholder="e.g., Technical, Communication, Leadership" />
                </Form.Item>

                <Form.Item
                  name="proficiencyLevel"
                  label="Proficiency Level"
                  rules={[
                    {
                      required: true,
                      message: "Proficiency Level is required",
                    },
                  ]}
                >
                  <Select
                    options={proficiencyLevelOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </Form.Item>

                <Form.Item name="lastAssessedDate" label="Last Assessed Date">
                  <DatePicker format="YYYY-MM-DD" className="w-full" />
                </Form.Item>

                <Form.Item name="remarks" label="Remarks">
                  <Input.TextArea rows={2} />
                </Form.Item>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
