"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
  SearchOutlined,
  FileTextOutlined,
  CheckCircleTwoTone,
  FileDoneOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { motion } from "framer-motion";
import { toast } from "sonner";
import { useRouter } from "next/navigation";
import {
  createMinimumQualification,
  updateMinimumQualification,
  deleteMinimumQualification,
  getAllMinimumQualifications,
  approveOrRejectMinimumQualification,
  MinimumQualification,
} from "@/app/api/minimumqualificationcriteria/minimumqualificationcriteria";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../../components/common/GlassCard";
import isBetween from "dayjs/plugin/isBetween";
dayjs.extend(isBetween);
import { highlightMatch } from "../../components/common/HighLightMatch";
import { FaTrash } from "react-icons/fa";
import { FileCheck, FileText, FileClock } from "lucide-react";
import { BsMortarboardFill } from "react-icons/bs";
import { LiaUserEditSolid } from "react-icons/lia";
import * as XLSX from "xlsx";
import { saveAs } from "file-saver";
import jsPDF from "jspdf";
import autoTable from "jspdf-autotable";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import { FaFileExcel, FaFilePdf, FaFileExport } from "react-icons/fa";

const WidgetCard = dynamic(() => import("../../components/common/WidgetCard"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading widget...</p>,
});

const ColumnVisibilityToggle = dynamic(
  () => import("../../components/common/ManageColumn"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
  }
);

const ConfirmDialog = dynamic(
  () => import("../../components/common/ConfirmDialog"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
  }
);

const { RangePicker } = DatePicker;

const statusOptions = ["Draft", "Submitted", "Approved", "Rejected"] as const;

export default function MinimumQualificationCriteria() {
  const [form] = Form.useForm();
  const [approvalForm] = Form.useForm();
  const [qualifications, setQualifications] = useState<MinimumQualification[]>(
    []
  );
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingQualification, setEditingQualification] =
    useState<MinimumQualification | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });
  const [loading, setLoading] = useState(false);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    status: undefined as string | undefined,
  });

  // Approval Modal States
  const [isApprovalModalOpen, setIsApprovalModalOpen] = useState(false);
  const [editingApprovalRecord, setEditingApprovalRecord] =
    useState<MinimumQualification | null>(null);
  const [forceRender, setForceRender] = useState(0);

  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();
  const router = useRouter();

  useEffect(() => {
    const fetchQualifications = async () => {
      try {
        setLoading(true);
        const data = await getAllMinimumQualifications();

        // Sort newest first on page load
        const sorted = data.sort((a: any, b: any) => {
          const dateA = a.created_at || "";
          const dateB = b.created_at || "";
          return dateB.localeCompare(dateA);
        });

        setQualifications(sorted);
      } catch (error) {
        console.error("Failed to fetch qualifications:", error);
      } finally {
        setLoading(false);
      }
    };
    fetchQualifications();
  }, []);

  const openAddModal = () => {
    setEditingQualification(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (qual: MinimumQualification) => {
    if (
      qual.status === "Submitted" ||
      qual.status === "Approved" ||
      qual.status === "Rejected"
    ) {
      // No toast here — backend will show if needed
      return;
    }

    setEditingQualification(qual);
    form.resetFields();
    form.setFieldsValue({
      criteria_code: qual.criteria_code,
      designation: qual.designation,
      department: qual.department,
      minimum_education: qual.minimum_education,
      minimum_experience: qual.minimum_experience,
      special_skills: qual.special_skills || "",
      specific_knowledge: qual.specific_knowledge || "",
      certifications_required: qual.certifications_required || "",
      remarks: qual.remarks || "",
      status: qual.status,
    });
    setIsModalOpen(true);
  };

  const openApprovalModal = (record: MinimumQualification) => {
    setEditingApprovalRecord(record);
    approvalForm.resetFields();

    const initialFields = {
      status: record.status || "Submitted",
      rejection_reason:
        record.status === "Rejected" ? record.rejection_reason || "" : "",
      approved_by_name:
        record.status === "Approved"
          ? record.approved_by_name || "Shankar.S"
          : "",
      approved_date:
        record.status === "Approved" && record.updated_at
          ? dayjs(record.updated_at)
          : undefined,
    };

    approvalForm.setFieldsValue(initialFields);
    setForceRender((prev) => prev + 1);
    setIsApprovalModalOpen(true);
  };

  const handleDelete = (criteria_code: string) => {
    const qualification = qualifications.find(
      (q) => q.criteria_code === criteria_code
    );

    if (!qualification) return;

    if (
      qualification.status === "Approved" ||
      qualification.status === "Rejected"
    ) {
      return; // Backend will show error toast
    }

    setConfirmDialog({
      open: true,
      title: "Delete Minimum Qualification",
      message: "Are you sure you want to delete this qualification criteria?",
      onConfirm: async () => {
        try {
          await deleteMinimumQualification(criteria_code);
          // NEW — New record appears at the TOP
          const updated = await getAllMinimumQualifications();
          const sorted = updated.sort((a: any, b: any) => {
            const dateA = a.created_at || "";
            const dateB = b.created_at || "";
            return dateB.localeCompare(dateA); // Newest first
          });
          setQualifications(sorted);
          // Success toast comes from API file
        } catch {
          // Error already shown by API file
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;

    setConfirmDialog({
      open: true,
      title: "Delete Multiple Qualifications",
      message: `Delete ${selectedRows.length} selected qualification(s)?`,
      onConfirm: async () => {
        try {
          await Promise.all(
            selectedRows.map((id) => deleteMinimumQualification(id))
          );
          const updated = await getAllMinimumQualifications();
          setQualifications(updated);
          setSelectedRows([]);
        } catch {
          // Error toast already shown in API
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();

      const payload: MinimumQualification = {
        designation: values.designation,
        department: values.department,
        minimum_education: values.minimum_education,
        minimum_experience: values.minimum_experience,
        special_skills: values.special_skills || "",
        specific_knowledge: values.specific_knowledge || "",
        certifications_required: values.certifications_required || "",
        remarks: values.remarks || "",
        status: values.status || "Draft",
      };

      if (editingQualification?.criteria_code) {
        await updateMinimumQualification(
          editingQualification.criteria_code,
          payload
        );
      } else {
        await createMinimumQualification(payload);
      }

      // Refresh list
      const updated = await getAllMinimumQualifications();
      setQualifications(updated);

      // Close modal
      setIsModalOpen(false);
      form.resetFields();
      setEditingQualification(null);

      // Success toast comes from API file
    } catch (error: any) {
      // Only handle frontend validation (required fields)
      if (error.errorFields && error.errorFields.length > 0) {
        const missingFields = error.errorFields.map((field: any) => {
          const fieldName = field.name[0];
          const labelMap: Record<string, string> = {
            designation: "Designation",
            department: "Department",
            minimum_education: "Minimum Education",
            minimum_experience: "Minimum Experience",
          };
          return labelMap[fieldName] || fieldName;
        });

        const message =
          missingFields.length > 1
            ? `${missingFields.join(", ")} are required`
            : `${missingFields[0]} is required`;

        // Still show frontend validation error
        toast.error(message);
        return;
      }

      // Backend errors are already shown in API file → do nothing here
    }
  };

  const handleApprovalSubmit = async () => {
    if (!editingApprovalRecord) return;

    try {
      const values = await approvalForm.validateFields();

      await approveOrRejectMinimumQualification(
        editingApprovalRecord.criteria_code!,
        values.status as "Approved" | "Rejected" | "Submitted",
        values.status === "Rejected" ? values.rejection_reason : undefined
      );

      const updated = await getAllMinimumQualifications();
      setQualifications(updated);
      setIsApprovalModalOpen(false);
      approvalForm.resetFields();
      // Success toast comes from API
    } catch {
      // Error toast already shown in API
    }
  };

  const renderStatusTag = (status: string) => {
    const colorMap: Record<string, string> = {
      Draft: "orange",
      Submitted: "blue",
      Approved: "green",
      Rejected: "red",
    };
    const iconMap: Record<string, React.ReactNode> = {
      Draft: <MinusCircleOutlined className="w-4 h-4" />,
      Submitted: <FileTextOutlined className="w-4 h-4" />,
      Approved: <CheckCircleTwoTone twoToneColor="#52c41a" />,
      Rejected: <CloseCircleOutlined />,
    };
    return (
      <Tag icon={iconMap[status]} color={colorMap[status] || "default"}>
        {status}
      </Tag>
    );
  };

  const filteredQualifications = qualifications.filter((q) => {
    const matchesSearch =
      q.designation?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      q.department?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      q.minimum_education?.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (q.created_at &&
        dayjs(q.created_at).isBetween(dateRange[0], dateRange[1], "day", "[]"));
    const matchesStatus =
      !advancedFilters.status || q.status === advancedFilters.status;
    return matchesSearch && matchesDate && matchesStatus;
  });

  const exportExcel = () => {
    const data = filteredQualifications.map((q) => ({
      "Criteria Code": q.criteria_code,
      Designation: q.designation,
      Department: q.department,
      "Minimum Education": q.minimum_education,
      "Minimum Experience": q.minimum_experience,
      "Special Skills": q.special_skills || "-",
      "Specific Knowledge": q.specific_knowledge || "-",
      "Certifications Required": q.certifications_required || "-",
      Remarks: q.remarks || "-",
      Status: q.status,
      "Created At": q.created_at
        ? dayjs(q.created_at).format("DD-MM-YYYY")
        : "-",
    }));

    const ws = XLSX.utils.json_to_sheet(data);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, "Qualification Criteria");

    const excelBuffer = XLSX.write(wb, { bookType: "xlsx", type: "array" });

    saveAs(
      new Blob([excelBuffer], { type: "application/octet-stream" }),
      "Minimum_Qualification_Criteria.xlsx"
    );

    toast.success("Excel exported successfully!");
  };

  const exportPDF = () => {
    const doc = new jsPDF("landscape");
    doc.setFontSize(14);
    doc.text("Minimum Qualification Criteria Report", 14, 10);

    const tableRows = filteredQualifications.map((q) => [
      q.criteria_code ?? "-",
      q.designation ?? "-",
      q.department ?? "-",
      q.minimum_education ?? "-",
      q.minimum_experience ?? "-",
      q.special_skills || "-",
      q.specific_knowledge || "-",
      q.certifications_required || "-",
      q.status ?? "-",
      q.created_at ? dayjs(q.created_at).format("DD-MM-YYYY") : "-",
    ]);

    autoTable(doc, {
      head: [
        [
          "Code",
          "Designation",
          "Department",
          "Min Education",
          "Experience",
          "Skills",
          "Knowledge",
          "Certifications",
          "Status",
          "Created At",
        ],
      ],
      body: tableRows,
      startY: 20,
      styles: { fontSize: 8 },
      headStyles: { fillColor: [52, 152, 219] },
    });

    doc.save("Minimum_Qualification_Criteria.pdf");
    toast.success("PDF exported successfully!");
  };

  const handleExport = (type: "excel" | "pdf") => {
    if (type === "excel") exportExcel();
    if (type === "pdf") exportPDF();
  };

  const paginatedData = filteredQualifications.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<MinimumQualification> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) =>
            setSelectedRows(
              e.target.checked
                ? filteredQualifications
                    .map((q) => q.criteria_code!)
                    .filter(Boolean)
                : []
            )
          }
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.criteria_code!)}
          onChange={(e) =>
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.criteria_code!]
                : prev.filter((id) => id !== record.criteria_code)
            )
          }
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Designation",
      dataIndex: "designation",
      key: "designation",
      sorter: (a, b) =>
        (a.designation || "").localeCompare(b.designation || ""),
      render: (text) => highlightMatch(text || "", searchTerm),
    },
    {
      title: "Department",
      dataIndex: "department",
      key: "department",
      sorter: (a, b) => (a.department || "").localeCompare(b.department || ""),
    },
    {
      title: "Min. Education",
      dataIndex: "minimum_education",
      key: "minimum_education",
    },
    {
      title: "Min. Experience",
      dataIndex: "minimum_experience",
      key: "minimum_experience",
      render: (val) => (val ? `${val} year${val === "1" ? "" : "s"}` : "-"),
    },
    // Status then the 4 requested fields in the table (showing full text)
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      render: renderStatusTag,
      filters: statusOptions.map((s) => ({ text: s, value: s })),
      onFilter: (value, record) => record.status === value,
    },
    {
      title: "Special Skills",
      dataIndex: "special_skills",
      key: "special_skills",
      render: (text) => (text ? String(text) : "-"),
    },
    {
      title: "Specific Knowledge",
      dataIndex: "specific_knowledge",
      key: "specific_knowledge",
      render: (text) => (text ? String(text) : "-"),
    },
    {
      title: "Certifications Required",
      dataIndex: "certifications_required",
      key: "certifications_required",
      render: (text) => (text ? String(text) : "-"),
    },
    {
      title: "Remarks",
      dataIndex: "remarks",
      key: "remarks",
      render: (text) => (text ? String(text) : "-"),
    },
    {
      title: "Created By",
      dataIndex: "created_by_name",
      key: "created_by_name",
      render: (text) => text || "-",
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      sorter: (a, b) => dayjs(a.created_at).unix() - dayjs(b.created_at).unix(),
      render: (value) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY")} (${dayjs(value).fromNow()})`
          : "-",
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Edit Approval">
            <Button
              icon={<LiaUserEditSolid />}
              onClick={() => openApprovalModal(record)}
              size="small"
              className="text-purple-600 border-purple-600 hover:bg-purple-50"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.criteria_code!)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "designation", label: "Designation" },
    { key: "department", label: "Department" },
    { key: "minimum_education", label: "Min. Education" },
    { key: "minimum_experience", label: "Min. Experience" },
    { key: "status", label: "Status" },
    { key: "special_skills", label: "Special Skills" },
    { key: "specific_knowledge", label: "Specific Knowledge" },
    { key: "certifications_required", label: "Certifications Required" },
    { key: "remarks", label: "Remarks" },
    { key: "created_by_name", label: "Created By" },
    { key: "created_at", label: "Created At" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);
  const handleVisibleColumnChange = useCallback(
    (cols: string[]) => setVisibleColumnKeys(cols),
    []
  );

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const QualificationCard = ({ qual }: { qual: MinimumQualification }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="success">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full space-y-2">
        <div className="flex justify-between items-start mb-2">
          <h3 className="text-lg font-semibold text-gray-800 dark:text-white">
            {qual.designation}
          </h3>
          {renderStatusTag(qual.status)}
        </div>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Department:</strong> {qual.department}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Min. Education:</strong> {qual.minimum_education}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Min. Experience:</strong> {qual.minimum_experience} year
          {qual.minimum_experience === "1" ? "" : "s"}
        </p>

        {/* Added full fields to grid card */}
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Special Skills:</strong>{" "}
          {qual.special_skills ? qual.special_skills : "-"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Specific Knowledge:</strong>{" "}
          {qual.specific_knowledge ? qual.specific_knowledge : "-"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Certifications Required:</strong>{" "}
          {qual.certifications_required ? qual.certifications_required : "-"}
        </p>
        <p className="text-sm text-gray-600 dark:text-gray-300">
          <strong>Remarks:</strong> {qual.remarks ? qual.remarks : "-"}
        </p>

        <p className="text-xs text-gray-500 dark:text-gray-400 mt-2">
          <strong>Created:</strong>{" "}
          {qual.created_at ? dayjs(qual.created_at).format("DD-MM-YYYY") : "-"}
        </p>
        <div className="flex gap-2 pt-2 border-t dark:border-gray-700">
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(qual)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Edit Approval">
            <Button
              icon={<LiaUserEditSolid />}
              onClick={() => openApprovalModal(qual)}
              size="small"
              className="text-purple-600 border-purple-600 hover:bg-purple-50"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(qual.criteria_code!)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </div>
      </div>
    </GlassCard>
  );

  const total = qualifications.length;
  const approved = qualifications.filter((q) => q.status === "Approved").length;
  const submitted = qualifications.filter(
    (q) => q.status === "Submitted"
  ).length;
  const draft = qualifications.filter((q) => q.status === "Draft").length;

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <BsMortarboardFill className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Minimum Qualification Criteria</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({qualifications.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Define minimum qualification requirements for designations.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/hrm",
            },
            {
              title: "Master Data",
            },
            {
              title: "Minimum Qualification Criteria",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Criteria"
            value={total}
            icon={<AppstoreOutlined />}
            color="blue"
            percentage={10}
            description="All qualification criteria"
          />
          <WidgetCard
            title="Approved"
            value={approved}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="green"
            percentage={5}
            description="Approved criteria"
          />
          <WidgetCard
            title="Submitted"
            value={submitted}
            icon={<FileTextOutlined className="w-5 h-5" />}
            color="blue"
            percentage={3}
            description="Awaiting approval"
          />
          <WidgetCard
            title="Draft"
            value={draft}
            icon={<FileClock className="w-5 h-5" />}
            color="orange"
            percentage={2}
            description="Draft entries"
          />
        </div>

        <div className="mt-4">
          <Card
            title="Minimum Qualification List"
            className="p-0 mb-6 shadow-sm"
          >
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search designation, department..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64"
                />

                <div className="flex gap-3">
                  {/* 🔥 EXPORT BUTTON GOES HERE */}
                  <Dropdown
                    menu={{
                      items: [
                        {
                          key: "excel",
                          label: (
                            <span className="flex items-center gap-2">
                              <FaFileExcel className="text-green-600" /> Export
                              Excel
                            </span>
                          ),
                        },
                        {
                          key: "pdf",
                          label: (
                            <span className="flex items-center gap-2">
                              <FaFilePdf className="text-red-600" /> Export PDF
                            </span>
                          ),
                        },
                      ],
                      onClick: (info: { key: string }) =>
                        handleExport(info.key as "excel" | "pdf"),
                    }}
                  >
                    <Button icon={<FaFileExport />}>Export</Button>
                  </Dropdown>

                  {/* ADD CRITERIA BUTTON */}
                  <Button
                    type="primary"
                    icon={<PlusOutlined />}
                    onClick={openAddModal}
                  >
                    Add Criteria
                  </Button>
                </div>
              </div>

              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    allowClear
                    onChange={(v) =>
                      setAdvancedFilters((prev) => ({ ...prev, status: v }))
                    }
                  >
                    {statusOptions.map((s) => (
                      <Select.Option key={s} value={s}>
                        {s}
                      </Select.Option>
                    ))}
                  </Select>
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="minqual_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(d) => setDateRange(d as any)}
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    icon={<FaTrash />}
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                    />
                  </div>
                </div>
              </div>
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedData.map((q) => (
                  <QualificationCard key={q.criteria_code} qual={q} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<MinimumQualification>
                  columns={filteredColumns}
                  dataSource={paginatedData}
                  rowKey="criteria_code"
                  scrollX="max-content"
                  className="mt-2"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}

            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(
                    currentPage * pageSize,
                    filteredQualifications.length
                  )}
                </strong>{" "}
                of <strong>{filteredQualifications.length}</strong> criteria
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredQualifications.length}
                onChange={(p, s) => {
                  setCurrentPage(p);
                  setPageSize(s);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        {/* Add/Edit Modal */}
        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <div className="flex items-center gap-1">
                  <FileCheck
                    className={`w-5 h-5 ${
                      !editingQualification
                        ? "text-green-600"
                        : "text-green-600"
                    }`}
                  />
                  <FileDoneOutlined
                    className={`w-5 h-5 ${
                      editingQualification ? "text-blue-600" : "text-gray-400"
                    }`}
                  />
                </div>
                <span>
                  {editingQualification ? "Edit Criteria" : "Add Criteria"}
                </span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            destroyOnHidden
            width="95vw"
            className="!top-5 font-semibold"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" },
            }}
          >
            <Form form={form} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Qualification Details
                  </h3>
                </div>

                {/* Show Criteria Code only in Edit mode */}
                {editingQualification && (
                  <Form.Item
                    name="criteria_code"
                    label="Criteria Code"
                    rules={[{ required: true }]}
                  >
                    <Input readOnly style={{ background: "#f3f3f3" }} />
                  </Form.Item>
                )}

                <Form.Item
                  name="designation"
                  label="Designation"
                  rules={[
                    { required: true, message: "Designation is required" },
                  ]}
                >
                  <Input />
                </Form.Item>
                <Form.Item
                  name="department"
                  label="Department"
                  rules={[
                    { required: true, message: "Department is required" },
                  ]}
                >
                  <Input />
                </Form.Item>
                <Form.Item
                  name="minimum_education"
                  label="Minimum Education"
                  rules={[{ required: true, message: "Education is required" }]}
                >
                  <Input />
                </Form.Item>

                <Form.Item
                  name="minimum_experience"
                  label="Minimum Experience (Years)"
                  rules={[
                    { required: true, message: "Experience is required" },
                  ]}
                >
                  <Input type="number" min={0} />
                </Form.Item>

                <Form.Item name="special_skills" label="Special Skills">
                  <Input.TextArea rows={2} />
                </Form.Item>
                <Form.Item name="specific_knowledge" label="Specific Knowledge">
                  <Input.TextArea rows={2} />
                </Form.Item>
                <Form.Item
                  name="certifications_required"
                  label="Certifications Required"
                >
                  <Input.TextArea rows={2} />
                </Form.Item>
                <Form.Item name="status" label="Status" initialValue="Draft">
                  <Select>
                    <Select.Option value="Draft">Draft</Select.Option>
                    <Select.Option value="Submitted">Submitted</Select.Option>
                  </Select>
                </Form.Item>

                <div className="col-span-full">
                  <Form.Item name="remarks" label="Remarks">
                    <Input.TextArea rows={3} />
                  </Form.Item>
                </div>
              </div>
            </Form>
          </Modal>
        </motion.div>

        {/* Edit Approval Modal */}
        <motion.div
          key={forceRender}
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                <IdcardOutlined className="w-5 h-5 text-purple-600" />
                <span>Edit Approval Status</span>
              </div>
            }
            open={isApprovalModalOpen}
            onOk={handleApprovalSubmit}
            onCancel={() => {
              setIsApprovalModalOpen(false);
              approvalForm.resetFields();
              setForceRender((prev) => prev + 1);
            }}
            destroyOnHidden
            width="95vw"
            className="!top-5 font-semibold"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" },
            }}
          >
            <Form form={approvalForm} layout="vertical" className="w-full px-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-4">
                <div className="col-span-full">
                  <h3 className="text-lg font-semibold mb-2">
                    Approval Details
                  </h3>
                </div>

                <Form.Item
                  name="status"
                  label="Approval Status"
                  rules={[{ required: true, message: "Status is required" }]}
                >
                  <Select
                    options={[
                      { label: "Submitted", value: "Submitted" },
                      { label: "Approved", value: "Approved" },
                      { label: "Rejected", value: "Rejected" },
                    ]}
                    onChange={(value) => {
                      const fields: any = { status: value };
                      if (value === "Rejected") {
                        fields.rejection_reason =
                          approvalForm.getFieldValue("rejection_reason") ||
                          editingApprovalRecord?.rejection_reason ||
                          "";
                      } else {
                        fields.rejection_reason = undefined;
                      }
                      if (value === "Approved") {
                        fields.approved_by_name = "Shankar.S";
                        fields.approved_date = dayjs();
                      } else {
                        fields.approved_by_name = undefined;
                        fields.approved_date = undefined;
                      }
                      approvalForm.setFieldsValue(fields);
                      setForceRender((prev) => prev + 1);
                    }}
                  />
                </Form.Item>

                {approvalForm.getFieldValue("status") === "Rejected" && (
                  <Form.Item
                    name="rejection_reason"
                    label="Rejection Reason"
                    rules={[
                      {
                        required: true,
                        message: "Rejection reason is required",
                      },
                    ]}
                  >
                    <Input.TextArea rows={3} />
                  </Form.Item>
                )}

                {approvalForm.getFieldValue("status") === "Approved" && (
                  <>
                    <Form.Item name="approved_by_name" label="Approved By">
                      <Input disabled />
                    </Form.Item>
                    <Form.Item name="approved_date" label="Approved Date">
                      <DatePicker disabled className="w-full" />
                    </Form.Item>
                  </>
                )}
              </div>
            </Form>
          </Modal>
        </motion.div>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
