"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import InputNumber from "antd/es/input-number";
import "antd/es/input-number/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  WarningOutlined,
  EyeOutlined,
  SearchOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
  FileTextOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useRouter } from "next/navigation";
import {
  createAppraisal,
  getAllAppraisals,
  getAppraisalByCode,
  updateAppraisal,
  approveAppraisal,
  deleteAppraisal,
  AppraisalRecord,
  CreateAppraisalData,
  UpdateAppraisalData,
  ApproveAppraisalData,
} from "@/app/api/performanceforappraisal/performanceforappraisal";
import { getAllEmployees, Employee } from "@/app/api/employee/employee";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../../components/common/GlassCard";
import { highlightMatch } from "../../components/common/HighLightMatch";
import { FaFileExcel, FaFileExport, FaFilePdf, FaTrash } from "react-icons/fa";
import { UserCheck, Users, Star } from "lucide-react";
import { LiaUserEditSolid } from "react-icons/lia";
import { useAuth } from "@/context/AuthContext";

const WidgetCard = dynamic(() => import("../../components/common/WidgetCard"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading widget...</p>,
});

const ColumnVisibilityToggle = dynamic(
  () => import("../../components/common/ManageColumn"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
  }
);

const ConfirmDialog = dynamic(
  () => import("../../components/common/ConfirmDialog"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
  }
);

const { RangePicker } = DatePicker;

const { Option } = Select;

export default function PerformanceForAppraisalPage() {
  const { user } = useAuth();
  const [form] = Form.useForm();
  const [approvalForm] = Form.useForm();
  const [appraisals, setAppraisals] = useState<AppraisalRecord[]>([]);
  const [employees, setEmployees] = useState<Employee[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [isApprovalModalOpen, setIsApprovalModalOpen] = useState(false);
  const [editingAppraisal, setEditingAppraisal] =
    useState<AppraisalRecord | null>(null);
  const [approvingAppraisal, setApprovingAppraisal] =
    useState<AppraisalRecord | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });
  const [loading, setLoading] = useState(false);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [forceRender, setForceRender] = useState(0);
  const isMobile = useIsMobile();
  const router = useRouter();

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        const [appraisalData, employeeData] = await Promise.all([
          getAllAppraisals(),
          getAllEmployees(),
        ]);

        const sortedAppraisals = appraisalData.sort((a, b) =>
          b.created_at.localeCompare(a.created_at)
        );

        setAppraisals(sortedAppraisals);
        setEmployees(employeeData);
      } catch (error) {
        toast.error("Failed to fetch appraisals or employees");
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  const openAddModal = () => {
    setEditingAppraisal(null);
    form.resetFields();
    setIsModalOpen(true);
  };

  const openEditModal = (record: AppraisalRecord) => {
    setEditingAppraisal(record);
    form.setFieldsValue({
      employee: record.employee_code,
      appraisalPeriod: record.appraisal_period,
      hrReviewer: record.hr_reviewer_id,
      qualityOfWork: record.quality_of_work,
      onTimeDelivery: record.on_time_delivery,
      teamwork: record.teamwork,
      innovation: record.innovation,
      leadership: record.leadership,
      remarks: record.remarks,
      hrRemarks: record.hr_remarks,
      newTrainingRecommended: record.new_training_recommended,
    });
    setIsModalOpen(true);
  };

  const openApprovalModal = (record: AppraisalRecord) => {
    setApprovingAppraisal(record);
    approvalForm.resetFields();
    setIsApprovalModalOpen(true);
  };

  const handleDelete = (code: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Appraisal",
      message: "Are you sure you want to delete this appraisal?",
      onConfirm: async () => {
        try {
          await deleteAppraisal(code);
          const updated = await getAllAppraisals();
          setAppraisals(
            updated.sort((a, b) => b.created_at.localeCompare(a.created_at))
          );
          toast.success("Appraisal deleted successfully");
        } catch {
          toast.error("Failed to delete appraisal");
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Appraisals",
      message: `Are you sure you want to delete ${selectedRows.length} appraisals?`,
      onConfirm: async () => {
        try {
          await Promise.all(selectedRows.map(deleteAppraisal));
          const updated = await getAllAppraisals();
          setAppraisals(
            updated.sort((a, b) => b.created_at.localeCompare(a.created_at))
          );
          setSelectedRows([]);
          toast.success("Appraisals deleted successfully");
        } catch {
          toast.error("Failed to delete selected appraisals");
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();

      const payload: CreateAppraisalData | UpdateAppraisalData = {
        employee_code: values.employee,
        appraisal_period: values.appraisalPeriod,
        hr_reviewer_id: values.hrReviewer,
        quality_of_work: values.qualityOfWork,
        on_time_delivery: values.onTimeDelivery,
        teamwork: values.teamwork,
        innovation: values.innovation,
        leadership: values.leadership,
        remarks: values.remarks,
        hr_remarks: values.hrRemarks,
        new_training_recommended: values.newTrainingRecommended,
      };

      if (editingAppraisal) {
        await updateAppraisal(
          editingAppraisal.appraisal_code,
          payload as UpdateAppraisalData
        );
        toast.success("Appraisal updated successfully");
      } else {
        await createAppraisal(payload as CreateAppraisalData);
        toast.success("Appraisal created successfully");
      }

      const updated = await getAllAppraisals();
      setAppraisals(
        updated.sort((a, b) => b.created_at.localeCompare(a.created_at))
      );
      setIsModalOpen(false);
      form.resetFields();
      setEditingAppraisal(null);
    } catch (error: any) {
      if (error.errorFields) {
        toast.error("Please fill all required fields");
      } else {
        toast.error(error.message || "Failed to save appraisal");
      }
    }
  };

  const handleApprovalSubmit = async () => {
    if (!approvingAppraisal) return;

    try {
      // Get correct user id stored during login
      const approverId = user?.user_id || localStorage.getItem("userId");

      console.log("Fetched Approver ID →", approverId);
      if (!approverId) {
        toast.error("User ID missing. Please login again.");

        return;
      }

      console.log("Approver ID Sent →", approverId);

      await approveAppraisal(approvingAppraisal.appraisal_code, {
        approver_id: approverId,
      });

      toast.success("Appraisal approved successfully");

      const updated = await getAllAppraisals();
      setAppraisals(
        updated.sort((a, b) => b.created_at.localeCompare(a.created_at))
      );

      setIsApprovalModalOpen(false);
    } catch (error: any) {
      const msg =
        error?.response?.data?.detail ||
        error?.response?.data?.message ||
        "Approval failed";
      toast.error(msg);
    }
  };

  const renderRatingTag = (rating: number) => {
    const color = rating >= 80 ? "green" : rating >= 60 ? "orange" : "red";
    return <Tag color={color}>{rating}/100</Tag>;
  };

  const renderOverallRating = (rating: number) => {
    const color = rating >= 80 ? "green" : rating >= 60 ? "orange" : "red";
    const icon =
      rating >= 80 ? (
        <CheckCircleOutlined />
      ) : rating >= 60 ? (
        <PauseCircleOutlined />
      ) : (
        <CloseCircleOutlined />
      );
    return (
      <Tag icon={icon} color={color}>
        {rating}%
      </Tag>
    );
  };

  const filteredAppraisals = appraisals.filter((app) => {
    const matchesSearch =
      app.appraisal_code.toLowerCase().includes(searchTerm.toLowerCase()) ||
      app.employee_name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      app.appraisal_period.includes(searchTerm);
    const matchesDate =
      !dateRange ||
      (app.created_at &&
        dayjs(app.created_at).isAfter(dateRange[0]) &&
        dayjs(app.created_at).isBefore(dateRange[1]));
    return matchesSearch && matchesDate;
  });

  const paginatedAppraisals = filteredAppraisals.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<AppraisalRecord> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) =>
            setSelectedRows(
              e.target.checked
                ? filteredAppraisals.map((a) => a.appraisal_code)
                : []
            )
          }
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.appraisal_code)}
          onChange={(e) =>
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.appraisal_code]
                : prev.filter((id) => id !== record.appraisal_code)
            )
          }
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Appraisal Code",
      dataIndex: "appraisal_code",
      key: "appraisal_code",
      sorter: (a, b) => a.appraisal_code.localeCompare(b.appraisal_code),
    },
    {
      title: "Employee",
      dataIndex: "employee_name",
      key: "employee_name",
      render: (text) => highlightMatch(text || "-", searchTerm),
    },
    { title: "Period", dataIndex: "appraisal_period", key: "appraisal_period" },
    {
      title: "HR Reviewer",
      dataIndex: "hr_reviewer_name",
      key: "hr_reviewer_name",
    },
    {
      title: "Quality",
      dataIndex: "quality_of_work",
      key: "quality_of_work",
      render: renderRatingTag,
    },
    {
      title: "Delivery",
      dataIndex: "on_time_delivery",
      key: "on_time_delivery",
      render: renderRatingTag,
    },
    {
      title: "Teamwork",
      dataIndex: "teamwork",
      key: "teamwork",
      render: renderRatingTag,
    },
    {
      title: "Innovation",
      dataIndex: "innovation",
      key: "innovation",
      render: (v) => (v ? renderRatingTag(v) : "-"),
    },
    {
      title: "Leadership",
      dataIndex: "leadership",
      key: "leadership",
      render: (v) => (v ? renderRatingTag(v) : "-"),
    },
    {
      title: "Overall Rating",
      dataIndex: "overall_rating",
      key: "overall_rating",
      render: renderOverallRating,
      sorter: (a, b) => a.overall_rating - b.overall_rating,
    },
    {
      title: "Training Recommended",
      dataIndex: "new_training_recommended",
      key: "new_training_recommended",
      render: (v) => (v ? <Tag color="green">Yes</Tag> : <Tag>No</Tag>),
    },
    {
      title: "Approved By",
      dataIndex: "approved_by_name",
      key: "approved_by_name",
      render: (text) => text || "-",
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      render: (v) => dayjs(v).format("DD-MM-YYYY"),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              type="primary"
              size="small"
            />
          </Tooltip>
          {!record.approved_by_name && (
            <Tooltip title="Approve">
              <Button
                icon={<CheckCircleOutlined />}
                onClick={() => openApprovalModal(record)}
                type="default"
                size="small"
                className="text-green-600"
              />
            </Tooltip>
          )}
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.appraisal_code)}
              danger
              size="small"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "appraisal_code", label: "Appraisal Code" },
    { key: "employee_name", label: "Employee" },
    { key: "appraisal_period", label: "Period" },
    { key: "hr_reviewer_name", label: "HR Reviewer" },
    { key: "quality_of_work", label: "Quality" },
    { key: "on_time_delivery", label: "Delivery" },
    { key: "teamwork", label: "Teamwork" },
    { key: "innovation", label: "Innovation" },
    { key: "leadership", label: "Leadership" },
    { key: "overall_rating", label: "Overall Rating" },
    { key: "new_training_recommended", label: "Training Recommended" },
    { key: "approved_by_name", label: "Approved By" },
    { key: "created_at", label: "Created At" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);
  const handleVisibleColumnChange = useCallback(
    (cols: string[]) => setVisibleColumnKeys(cols),
    []
  );
  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const total = appraisals.length;
  const approved = appraisals.filter((a) => a.approved_by_name).length;
  const pending = total - approved;
  const highPerformers = appraisals.filter(
    (a) => a.overall_rating >= 80
  ).length;

  const handleExport = async (type: "pdf" | "excel") => {
    toast.info(`Export ${type.toUpperCase()} functionality coming soon!`);
  };

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <Star className="w-7 h-7 text-yellow-600 dark:text-yellow-400 flex-shrink-0" />
          <span>Performance Appraisal</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({total})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage employee performance appraisals and track ratings.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/hrm",
            },
            {
              title: "Performance Management",
            },
            {
              title: "Appraisals",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Appraisals"
            value={total}
            icon={<AppstoreOutlined />}
            color="blue"
            percentage={10}
            description="All appraisals"
          />
          <WidgetCard
            title="Approved"
            value={approved}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="green"
            percentage={8}
            description="Approved appraisals"
          />
          <WidgetCard
            title="Pending"
            value={pending}
            icon={<PauseCircleOutlined />}
            color="gold"
            percentage={-5}
            description="Pending approval"
          />
          <WidgetCard
            title="High Performers"
            value={highPerformers}
            icon={<FireOutlined />}
            color="orange"
            percentage={12}
            description="Overall ≥80%"
          />
        </div>

        <Card title="Appraisals List" className="p-0 mb-6">
          <div className="p-4 flex flex-col gap-4">
            <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
              <Input
                prefix={<SearchOutlined />}
                allowClear
                placeholder="Search by code, employee, period"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full sm:w-64"
              />
              <Dropdown
                menu={{
                  items: [
                    {
                      key: "pdf",
                      label: (
                        <span className="flex items-center gap-2">
                          <FaFilePdf className="text-red-500" /> Export PDF
                        </span>
                      ),
                    },
                    {
                      key: "excel",
                      label: (
                        <span className="flex items-center gap-2">
                          <FaFileExcel className="text-green-500" /> Export
                          Excel
                        </span>
                      ),
                    },
                  ],
                  onClick: ({ key }) => handleExport(key as "pdf" | "excel"),
                }}
              >
                <Button type="default" icon={<FaFileExport />}>
                  Export
                </Button>
              </Dropdown>
              <Button
                type="primary"
                icon={<PlusOutlined />}
                onClick={openAddModal}
                className="bg-blue-600 hover:bg-blue-700"
              >
                Add Appraisal
              </Button>
            </div>

            <div className="flex flex-col lg:flex-row justify-between gap-4 items-start lg:items-center">
              {/* LEFT SIDE — EMPTY OR KEEP FILTERS IF NEEDED */}
              <div className="w-full lg:w-auto flex flex-wrap gap-3">
                {/* Add left-side filter items here if needed */}
              </div>

              {/* RIGHT SIDE — PERFECTLY ALIGNED */}
              <div className="flex flex-wrap gap-3 items-center justify-start lg:justify-end w-full lg:w-auto">
                <ColumnVisibilityToggle
                  columns={columnKeys}
                  labels={labelsMap}
                  storageKey="appraisal_table_visibility"
                  onChange={handleVisibleColumnChange}
                />

                <RangePicker
                  value={dateRange}
                  onChange={(d) => setDateRange(d as any)}
                  className="min-w-[220px] w-full sm:w-auto"
                />

                <Button
                  onClick={handleBulkDelete}
                  disabled={selectedRows.length === 0}
                  danger
                  icon={<FaTrash />}
                  className="w-full sm:w-auto"
                >
                  Bulk Delete
                </Button>

                <Switch
                  checkedChildren={<AppstoreOutlined />}
                  unCheckedChildren={<TableOutlined />}
                  checked={isGridView}
                  onChange={setIsGridView}
                  className="ms-2"
                />
              </div>
            </div>
          </div>

          {isGridView || isMobile ? (
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
              {paginatedAppraisals.map((app) => (
                <GlassCard
                  key={app.appraisal_code}
                  gradient
                  blurStrength="lg"
                  hoverLift
                  color="success"
                >
                  <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all space-y-2">
                    <div className="flex justify-between items-start">
                      <h3 className="font-semibold">{app.employee_name}</h3>
                      {renderOverallRating(app.overall_rating)}
                    </div>
                    <p>
                      <strong>Code:</strong> {app.appraisal_code}
                    </p>
                    <p>
                      <strong>Period:</strong> {app.appraisal_period}
                    </p>
                    <p>
                      <strong>Quality:</strong>{" "}
                      {renderRatingTag(app.quality_of_work)}
                    </p>
                    <p>
                      <strong>Delivery:</strong>{" "}
                      {renderRatingTag(app.on_time_delivery)}
                    </p>
                    <p>
                      <strong>Teamwork:</strong> {renderRatingTag(app.teamwork)}
                    </p>
                    <div className="flex gap-2 pt-2 border-t dark:border-gray-700">
                      <Button
                        icon={<EditOutlined />}
                        onClick={() => openEditModal(app)}
                        size="small"
                        type="primary"
                      />
                      {!app.approved_by_name && (
                        <Button
                          icon={<CheckCircleOutlined />}
                          onClick={() => openApprovalModal(app)}
                          size="small"
                          className="text-green-600"
                        />
                      )}
                      <Button
                        icon={<DeleteOutlined />}
                        onClick={() => handleDelete(app.appraisal_code)}
                        danger
                        size="small"
                      />
                    </div>
                  </div>
                </GlassCard>
              ))}
            </div>
          ) : (
            <div className="overflow-auto">
              <DataTable
                columns={filteredColumns}
                dataSource={paginatedAppraisals}
                rowKey="appraisal_code"
                scrollX="max-content"
                pagination={false}
                loading={loading}
              />
            </div>
          )}

          <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
            <div className="mb-2 sm:mb-0">
              Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
              <strong>
                {Math.min(currentPage * pageSize, filteredAppraisals.length)}
              </strong>{" "}
              of <strong>{filteredAppraisals.length}</strong> appraisals
            </div>
            <Pagination
              current={currentPage}
              pageSize={pageSize}
              total={filteredAppraisals.length}
              onChange={(p, s) => {
                setCurrentPage(p);
                setPageSize(s);
              }}
              showSizeChanger
              pageSizeOptions={["10", "30", "50", "100"]}
              size="small"
            />
          </div>
        </Card>

        {/* Add/Edit Modal */}
        <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }}>
          <Modal
            title={
              <div className="flex items-center gap-3 border-b pb-2 border-gray-300 dark:border-gray-600">
                <Star className="w-6 h-6 text-yellow-600" />
                <span className="font-semibold text-lg">
                  {editingAppraisal ? "Edit Appraisal" : "Create Appraisal"}
                </span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            width="95vw"
            className="!top-6"
            styles={{ body: { maxHeight: "80vh", overflowY: "auto" } }}
          >
            <Form form={form} layout="vertical" className="mt-4 px-3">
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                {!editingAppraisal && (
                  <>
                    <Form.Item
                      name="employee"
                      label="Employee"
                      rules={[{ required: true }]}
                    >
                      <Select
                        showSearch
                        placeholder="Select employee"
                        options={employees.map((e) => ({
                          label: `${e.first_name} ${e.last_name || ""}`,
                          value: e.employee_code,
                        }))}
                      />
                    </Form.Item>

                    <Form.Item
                      name="appraisalPeriod"
                      label="Appraisal Period"
                      rules={[{ required: true }]}
                    >
                      <Input placeholder="e.g., 2024–2025" />
                    </Form.Item>

                    <Form.Item
                      name="hrReviewer"
                      label="HR Reviewer ID"
                      rules={[{ required: true }]}
                    >
                      <Input placeholder="Enter HR reviewer ID" />
                    </Form.Item>
                  </>
                )}

                <Form.Item
                  name="qualityOfWork"
                  label="Quality of Work"
                  rules={[{ required: true }]}
                >
                  <InputNumber min={1} max={100} className="w-full" />
                </Form.Item>

                <Form.Item
                  name="onTimeDelivery"
                  label="On-Time Delivery"
                  rules={[{ required: true }]}
                >
                  <InputNumber min={1} max={100} className="w-full" />
                </Form.Item>

                <Form.Item
                  name="teamwork"
                  label="Teamwork"
                  rules={[{ required: true }]}
                >
                  <InputNumber min={1} max={100} className="w-full" />
                </Form.Item>

                <Form.Item name="innovation" label="Innovation">
                  <InputNumber min={1} max={100} className="w-full" />
                </Form.Item>

                <Form.Item name="leadership" label="Leadership">
                  <InputNumber min={1} max={100} className="w-full" />
                </Form.Item>

                <Form.Item name="remarks" label="Employee Remarks">
                  <Input.TextArea
                    rows={2}
                    placeholder="Add employee remarks..."
                  />
                </Form.Item>

                <Form.Item name="hrRemarks" label="HR Remarks">
                  <Input.TextArea rows={2} placeholder="Add HR remarks..." />
                </Form.Item>

                <Form.Item
                  name="newTrainingRecommended"
                  label="Recommend New Training"
                  valuePropName="checked"
                >
                  <Switch />
                </Form.Item>
              </div>
            </Form>
          </Modal>
        </motion.div>

        {/* Approval Modal */}
        <Modal
          title="Approve Appraisal"
          open={isApprovalModalOpen}
          onOk={handleApprovalSubmit}
          onCancel={() => setIsApprovalModalOpen(false)}
          okText="Approve"
          width="500px"
        >
          <p>Are you sure you want to approve this appraisal?</p>
          <p>
            <strong>Employee:</strong> {approvingAppraisal?.employee_name}
          </p>
          <p>
            <strong>Period:</strong> {approvingAppraisal?.appraisal_period}
          </p>
          <p>
            <strong>Overall Rating:</strong>{" "}
            {approvingAppraisal &&
              renderOverallRating(approvingAppraisal.overall_rating)}
          </p>
        </Modal>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
