"use client";

import React, { useCallback, useEffect, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import Upload from "antd/es/upload";
import "antd/es/upload/style";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SearchOutlined,
  DownloadOutlined,
  WarningOutlined,
  CheckCircleOutlined,
  CloseCircleOutlined,
  FileOutlined,
  ShoppingCartOutlined,
  SendOutlined,
  PlusCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "@/app/components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import {
  type PurchaseOrder,
  getAllPurchaseOrders,
  createPurchaseOrder,
  updatePurchaseOrder,
  deletePurchaseOrderById,
  downloadPurchaseOrderDocument,
  PurchaseOrderCreate,
} from "@/app/api/purchaseorder/purchaseorder";
// import { getAllProducts, Product } from "@/app/api/products/product";
import { getAllSuppliers, Supplier } from "@/app/api/suppliers/suppliers";
import {
  getPurchaseRequisitions,
  PurchaseRequisition,
} from "@/app/api/purchaserequisition/purchaserequisition";
import ConfirmDialog from "@/app/components/common/ConfirmDialog";
import GlassCard from "@/app/components/common/GlassCard";
import { TruckIcon } from "lucide-react";
const WidgetCard = dynamic(() => import("@/app/components/common/WidgetCard"), {
  ssr: false,
});

const ColumnVisibilityToggle = dynamic(
  () => import("@/app/components/common/ManageColumn"),
  {
    ssr: false,
  }
);

const approveStatusOptions = ["Pending", "Approved", "Rejected"] as const;
const statusOptions = [
  "Draft",
  "Sent",
  "Confirmed",
  "Delivered",
  "Closed",
  "Cancelled",
] as const;

export default function PurchaseOrderPage() {
  const [form] = Form.useForm();
  const [purchaseOrders, setPurchaseOrders] = useState<PurchaseOrder[]>([]);
  // const [products, setProducts] = useState<Product[]>([]);
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [purchaseRequisitions, setPurchaseRequisitions] = useState<
    PurchaseRequisition[]
  >([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingPurchaseOrder, setEditingPurchaseOrder] =
    useState<PurchaseOrder | null>(null);
  const [viewingPurchaseOrder, setViewingPurchaseOrder] =
    useState<PurchaseOrder | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    Status: undefined,
    Approve_status: undefined,
    supplier_id: undefined,
  });
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [loading, setLoading] = useState(false);
  const [supportedFileList, setSupportedFileList] = useState<
    import("antd/es/upload/interface").UploadFile[]
  >([]);
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    title: "",
    message: "",
    onConfirm: () => {},
  });

  const isMobile = useIsMobile();

  // Fetch data on mount
  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      try {
        const [supplierData, requisitionData, purchaseOrderData] =
          await Promise.all([
            getAllSuppliers(),
            getPurchaseRequisitions(),
            getAllPurchaseOrders(),
          ]);

        setSuppliers(supplierData);
        setPurchaseRequisitions(requisitionData);
        setPurchaseOrders(purchaseOrderData ?? []);
      } catch (error) {
        toast.error("Failed to fetch data");
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  const openAddModal = () => {
    form.resetFields();
    setEditingPurchaseOrder(null);
    setSupportedFileList([]);
    setIsModalOpen(true);
  };

  const openEditModal = (purchaseOrder: PurchaseOrder) => {
    form.setFieldsValue({
      ...purchaseOrder,
      po_date: purchaseOrder.po_date ? dayjs(purchaseOrder.po_date) : undefined,
      expected_delivery_date: purchaseOrder.expected_delivery_date
        ? dayjs(purchaseOrder.expected_delivery_date)
        : undefined,
    });
    setEditingPurchaseOrder(purchaseOrder);
    setSupportedFileList(
      purchaseOrder.supported_file &&
        typeof purchaseOrder.supported_file === "string"
        ? [
            {
              uid: "-1",
              name: purchaseOrder.supported_file.split("/").pop() || "document",
              status: "done",
              url: purchaseOrder.supported_file,
            },
          ]
        : []
    );
    setIsModalOpen(true);
  };

  const handleDelete = (purchase_order_id: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Purchase Order",
      message: "Are you sure you want to delete this purchase order?",
      onConfirm: async () => {
        try {
          await deletePurchaseOrderById(purchase_order_id);
          toast.success("Purchase Order deleted successfully");
          setPurchaseOrders((prev) =>
            prev.filter((po) => po.purchase_order_id !== purchase_order_id)
          );
        } catch (error) {
          toast.error("Failed to delete Purchase Order");
        }
      },
    });
  };

  const handleBulkDelete = () => {
    if (selectedRows.length === 0) return;

    setConfirmDialog({
      open: true,
      title: "Delete Purchase Orders",
      message: `Are you sure you want to delete ${selectedRows.length} purchase orders?`,
      onConfirm: async () => {
        try {
          for (const purchase_order_id of selectedRows) {
            await deletePurchaseOrderById(purchase_order_id);
          }
          toast.success("Purchase Orders deleted successfully");
          setSelectedRows([]);
          setPurchaseOrders((prev) =>
            prev.filter((po) => !selectedRows.includes(po.purchase_order_id!))
          );
        } catch (error) {
          toast.error("Failed to delete Purchase Orders");
        }
      },
    });
  };

  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();
      const data: Partial<PurchaseOrder> = {
        ...values,
        po_date: values.po_date
          ? dayjs(values.po_date).format("YYYY-MM-DD")
          : undefined,
        expected_delivery_date: values.expected_delivery_date
          ? dayjs(values.expected_delivery_date).format("YYYY-MM-DD")
          : undefined,
        ...(supportedFileList[0]?.originFileObj && {
          supported_file: supportedFileList[0].originFileObj,
        }),
      };

      if (editingPurchaseOrder) {
        await updatePurchaseOrder(
          editingPurchaseOrder.purchase_order_id!,
          data
        );
        toast.success("Purchase Order updated successfully");
      } else {
        await createPurchaseOrder(data as PurchaseOrderCreate);
        toast.success("Purchase Order created successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
      setSupportedFileList([]);
      const purchaseOrderData = await getAllPurchaseOrders();
      setPurchaseOrders(purchaseOrderData || []);
    } catch (error) {
      toast.error("Failed to save purchase order");
    }
  };

  const handleDownloadDocument = async (purchase_order_id: string) => {
    try {
      await downloadPurchaseOrderDocument(purchase_order_id);
      toast.success("Purchase Order document downloaded");
    } catch (error) {
      toast.error("Failed to download Purchase Order document");
    }
  };

  const renderApproveStatusTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Pending: "gold",
      Approved: "green",
      Rejected: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Pending: <WarningOutlined />,
      Approved: <CheckCircleOutlined />,
      Rejected: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const renderStatusTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Draft: "gray",
      Sent: "blue",
      Confirmed: "green",
      Delivered: "cyan",
      Closed: "purple",
      Cancelled: "red",
    };

    return <Tag color={colorMap[value] || "default"}>{value}</Tag>;
  };

  const getSupplierDisplay = (supplier_id: string) => {
    const supplier = suppliers.find((s) => s.supplier_id === supplier_id);
    return supplier
      ? `${supplier.company_name} (${supplier.contact_person || "N/A"})`
      : supplier_id;
  };

  const getProductDisplay = (product_id: string) => {
    // If you have a products array, you can use it here.
    // For now, just return the product_id as a fallback.
    return product_id || "N/A";
  };

  const filteredPurchaseOrders = (purchaseOrders || []).filter((po) => {
    const supplier = suppliers.find((s) => s.supplier_id === po.supplier_id);
    // const product = products.find((p) => p.product_id === po.product_id);
    const matchesSearch =
      (supplier?.company_name
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      (supplier?.contact_person
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false) ||
      // (product?.product_id?.toLowerCase().includes(searchTerm.toLowerCase()) ??
      //   false) ||
      (po.purchase_order_id?.toLowerCase().includes(searchTerm.toLowerCase()) ??
        false) ||
      (po.purchase_requisition_id
        ?.toLowerCase()
        .includes(searchTerm.toLowerCase()) ??
        false);

    const matchesDate =
      !dateRange ||
      (po.created_at &&
        (!dateRange[0] ||
          dayjs(po.created_at).isSame(dateRange[0]) ||
          dayjs(po.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(po.created_at).isSame(dateRange[1]) ||
          dayjs(po.created_at).isBefore(dateRange[1])));

    const matchesAdvanced =
      (!advancedFilters.Status || po.Status === advancedFilters.Status) &&
      (!advancedFilters.Approve_status ||
        po.Approve_status === advancedFilters.Approve_status) &&
      (!advancedFilters.supplier_id ||
        po.supplier_id === advancedFilters.supplier_id);

    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedPurchaseOrders = filteredPurchaseOrders.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<PurchaseOrder> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(
              checked
                ? filteredPurchaseOrders.map((po) => po.purchase_order_id!)
                : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.purchase_order_id!)}
          onChange={(e) => {
            const rowId = record.purchase_order_id!;
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, rowId]
                : prev.filter((id) => id !== rowId)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "PO ID",
      dataIndex: "purchase_order_id",
      key: "purchase_order_id",
      sorter: (a, b) =>
        (a.purchase_order_id || "").localeCompare(b.purchase_order_id || ""),
      render: (text) => (
        <span className="text-blue-600 font-medium">{text}</span>
      ),
    },
    {
      title: "Requisition ID",
      dataIndex: "purchase_requisition_id",
      key: "purchase_requisition_id",
      sorter: (a, b) =>
        a.purchase_requisition_id.localeCompare(b.purchase_requisition_id),
    },
    {
      title: "Supplier Id",
      dataIndex: "supplier_id",
      key: "supplier_id",
      // render: (product_id) =>
      //   product_id ? getProductDisplay(product_id) : "N/A",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Supplier",
      dataIndex: "supplier_id",
      key: "supplier_id",
      sorter: (a, b) => {
        const supplierA = suppliers.find(
          (s) => s.supplier_id === a.supplier_id
        );
        const supplierB = suppliers.find(
          (s) => s.supplier_id === b.supplier_id
        );
        return (supplierA?.company_name || a.supplier_id).localeCompare(
          supplierB?.company_name || b.supplier_id
        );
      },
      render: (supplier_id) => (
        <span className="text-green-600 font-medium">
          {getSupplierDisplay(supplier_id)}
        </span>
      ),
    },
    {
      title: "Product Id",
      dataIndex: "product_id",
      key: "product_id",
      // render: (product_id) =>
      //   product_id ? getProductDisplay(product_id) : "N/A",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Product Name",
      dataIndex: "product_name",
      key: "product_name",
      // render: (product_id) =>
      //   product_id ? getProductDisplay(product_id) : "N/A",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "PO Date",
      dataIndex: "po_date",
      key: "po_date",
      sorter: (a, b) =>
        dayjs(a.po_date || "").unix() - dayjs(b.po_date || "").unix(),
      render: (text) => (text ? dayjs(text).format("YYYY-MM-DD") : "N/A"),
    },
    {
      title: "Expected Delivery",
      dataIndex: "expected_delivery_date",
      key: "expected_delivery_date",
      sorter: (a, b) =>
        dayjs(a.expected_delivery_date || "").unix() -
        dayjs(b.expected_delivery_date || "").unix(),
      render: (text) => (text ? dayjs(text).format("YYYY-MM-DD") : "N/A"),
    },
    {
      title: "Quantity",
      dataIndex: "quantity_ordered",
      key: "quantity_ordered",
      sorter: (a, b) => (a.quantity_ordered || 0) - (b.quantity_ordered || 0),
      // render: (text, record) =>
      //   text ? `${text} ${record.unit_of_measure || ""}` : "N/A",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Payment Terms",
      dataIndex: "payment_terms",
      key: "payment_terms",
      // sorter: (a, b) => (a.quantity_ordered || 0) - (b.quantity_ordered || 0),
      // render: (text, record) =>
      //   text ? `${text} ${record.unit_of_measure || ""}` : "N/A",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Unit_Of_Measure",
      dataIndex: "unit_of_measure",
      key: "unit_of_measure",
      // sorter: (a, b) => (a.quantity_ordered || 0) - (b.quantity_ordered || 0),
      // render: (text, record) =>
      //   text ? `${text} ${record.unit_of_measure || ""}` : "N/A",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Price/Unit",
      dataIndex: "price_per_unit",
      key: "price_per_unit",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Total Amount",
      dataIndex: "total_amount",
      key: "total_amount",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Remarks",
      dataIndex: "remarks",
      key: "remarks",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Approved At",
      dataIndex: "approved_at",
      key: "approved_at",
      render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Created By",
      dataIndex: "created_by_name",
      key: "created_by_name",
      // sorter: (a, b) => (a.quantity_ordered || 0) - (b.quantity_ordered || 0),
      render: (text, record) =>
        text ? `${text} (${record.created_by_id || ""})` : "N/A",
      // render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Approved By",
      dataIndex: "approved_by_name",
      key: "approved_by_name",
      // sorter: (a, b) => (a.quantity_ordered || 0) - (b.quantity_ordered || 0),
      render: (text, record) =>
        text ? `${text} (${record.approved_by_id || ""})` : "N/A",
      // render: (text) => (text ? `${text}` : "N/A"),
    },
    {
      title: "Approval Status",
      dataIndex: "Approve_status",
      key: "Approve_status",
      filters: approveStatusOptions.map((status) => ({
        text: status,
        value: status,
      })),
      onFilter: (value, record) => record.Approve_status === value,
      render: renderApproveStatusTag,
      sorter: (a, b) => a.Approve_status.localeCompare(b.Approve_status),
    },
    {
      title: "Status",
      dataIndex: "Status",
      key: "Status",
      filters: statusOptions.map((status) => ({ text: status, value: status })),
      onFilter: (value, record) => record.Status === value,
      render: renderStatusTag,
      sorter: (a, b) => a.Status.localeCompare(b.Status),
    },
    {
      title: "Document",
      key: "document",
      render: (_, record) =>
        record.supported_file ? (
          <Button
            icon={<DownloadOutlined />}
            onClick={() => handleDownloadDocument(record.purchase_order_id!)}
            size="small"
            className="text-green-600"
          >
            Download
          </Button>
        ) : (
          "N/A"
        ),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingPurchaseOrder(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.purchase_order_id!)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = columns.map((col) => ({
    key: col.key as string,
    label: col.title as string,
  }));

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const PurchaseOrderCard = ({
    purchaseOrder,
  }: {
    purchaseOrder: PurchaseOrder;
  }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="warning">
      <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
        {/* Header */}
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
          <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
            PO: {purchaseOrder.purchase_order_id || "N/A"}
          </h3>
          <div className="flex gap-1">
            {renderApproveStatusTag(purchaseOrder.Approve_status)}
            {renderStatusTag(purchaseOrder.Status)}
          </div>
        </div>

        {/* Main Info Grid */}
        <div className="grid grid-cols-1 gap-y-2 text-sm">
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Supplier:</strong>{" "}
            {purchaseOrder.supplier_name ||
              getSupplierDisplay(purchaseOrder.supplier_id)}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Product:</strong>{" "}
            {purchaseOrder.product_name ||
              (purchaseOrder.product_id
                ? getProductDisplay(purchaseOrder.product_id)
                : "N/A")}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Quantity:</strong>{" "}
            {purchaseOrder.quantity_ordered != null
              ? `${purchaseOrder.quantity_ordered} ${
                  purchaseOrder.unit_of_measure || ""
                }`
              : "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Price / Unit:</strong>{" "}
            {purchaseOrder.price_per_unit || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Total Amount:</strong>{" "}
            {purchaseOrder.total_amount
              ? `$${purchaseOrder.total_amount}`
              : "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Expected Delivery:</strong>{" "}
            {purchaseOrder.expected_delivery_date
              ? dayjs(purchaseOrder.expected_delivery_date).format("YYYY-MM-DD")
              : "N/A"}
          </p>

          {purchaseOrder.supported_file && (
            <div className="text-gray-600 dark:text-gray-300 break-words">
              <strong>Attachment:</strong>{" "}
              <span className="break-words">
                {typeof purchaseOrder.supported_file === "string"
                  ? purchaseOrder.supported_file
                  : purchaseOrder.supported_file.name}
              </span>
            </div>
          )}
        </div>

        {/* Actions */}
        <div className="mt-2 flex gap-2">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingPurchaseOrder(purchaseOrder)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(purchaseOrder)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(purchaseOrder.purchase_order_id!)}
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
          {purchaseOrder.supported_file && (
            <Tooltip title="Download">
              <Button
                icon={<DownloadOutlined />}
                onClick={() =>
                  handleDownloadDocument(purchaseOrder.purchase_order_id!)
                }
                size="small"
                className="text-green-600"
              >
                Download
              </Button>
            </Tooltip>
          )}
        </div>

        {/* Created/Updated */}
        <p className="text-gray-500 dark:text-gray-400 text-xs mt-2">
          <strong>Created:</strong>{" "}
          {purchaseOrder.created_at
            ? dayjs(purchaseOrder.created_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>
        <p className="text-gray-500 dark:text-gray-400 text-xs">
          <strong>Updated:</strong>{" "}
          {purchaseOrder.updated_at
            ? dayjs(purchaseOrder.updated_at).format("YYYY-MM-DD")
            : "N/A"}
        </p>
      </div>
    </GlassCard>
  );

  PurchaseOrderCard.displayName = "PurchaseOrderCard";

  const total = purchaseOrders.length;
  const pending = purchaseOrders.filter(
    (po) => po.Approve_status === "Pending"
  ).length;
  const approved = purchaseOrders.filter(
    (po) => po.Approve_status === "Approved"
  ).length;
  const draft = purchaseOrders.filter((po) => po.Status === "Draft").length;

  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <TruckIcon className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Purchase Orders</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({purchaseOrders.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage purchase orders and track procurement process.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Procurement",
            },
            {
              title: "Purchase Orders",
            },
          ]}
        />

        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Orders"
            value={total}
            icon={<ShoppingCartOutlined />}
            color="blue"
            percentage={12}
            description="All purchase orders"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,128,255,0.2)] dark:shadow-[0_0_10px_rgba(0,128,255,0.3)]"
          />
          <WidgetCard
            title="Pending Approval"
            value={pending}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="Awaiting approval"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(255,165,0,0.2)] dark:shadow-[0_0_10px_rgba(255,165,0,0.3)]"
          />
          <WidgetCard
            title="Approved Orders"
            value={approved}
            icon={<CheckCircleOutlined />}
            color="green"
            percentage={8}
            description="Approved orders"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(0,255,0,0.2)] dark:shadow-[0_0_10px_rgba(0,255,0,0.3)]"
          />
          <WidgetCard
            title="Draft Orders"
            value={draft}
            icon={<FileOutlined />}
            color="gray"
            percentage={3}
            description="Orders in draft"
            trendData={trendData}
            className="shadow-[0_0_10px_rgba(128,128,128,0.2)] dark:shadow-[0_0_10px_rgba(128,128,128,0.3)]"
          />
        </div>

        <div className="mt-4">
          <Card title="Purchase Order List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search PO ID, Supplier, or Product"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Purchase Order
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, Status: value }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Approval Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        Approve_status: value,
                      }))
                    }
                    allowClear
                    options={approveStatusOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                  <Select
                    placeholder="Supplier"
                    className="w-full max-w-[200px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        supplier_id: value,
                      }))
                    }
                    allowClear
                    options={suppliers.map((supplier) => ({
                      label: supplier.company_name,
                      value: supplier.supplier_id,
                    }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="purchase_order_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <DatePicker.RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.Status ||
                advancedFilters.Approve_status ||
                advancedFilters.supplier_id) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.Status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          Status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.Status}
                    </Tag>
                  )}
                  {advancedFilters.Approve_status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          Approve_status: undefined,
                        }))
                      }
                    >
                      Approval Status: {advancedFilters.Approve_status}
                    </Tag>
                  )}
                  {advancedFilters.supplier_id && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          supplier_id: undefined,
                        }))
                      }
                    >
                      Supplier:{" "}
                      {getSupplierDisplay(advancedFilters.supplier_id)}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedPurchaseOrders.map((purchaseOrder) => (
                  <PurchaseOrderCard
                    key={purchaseOrder.purchase_order_id}
                    purchaseOrder={purchaseOrder}
                  />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<PurchaseOrder>
                  columns={filteredColumns}
                  dataSource={paginatedPurchaseOrders}
                  rowKey={(record) => record.purchase_order_id!}
                  scrollX="max-content"
                  className="mt-4"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(
                    currentPage * pageSize,
                    filteredPurchaseOrders.length
                  )}
                </strong>{" "}
                of <strong>{filteredPurchaseOrders.length}</strong> purchase
                orders
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredPurchaseOrders.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <span className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                {editingPurchaseOrder ? (
                  <>
                    <EditOutlined className="w-5 h-5 text-blue-600" />
                    Edit Purchase Order
                  </>
                ) : (
                  <>
                    <PlusCircleOutlined className="w-5 h-5 text-green-600" />
                    Add Purchase Order
                  </>
                )}
              </span>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
              setSupportedFileList([]);
            }}
            destroyOnHidden
            width="95%" // responsive width
            className="!top-5 font-semibold"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line
            }}
            // okButtonProps={{ loading: isSubmitting }}
          >
            <Form
              form={form}
              layout="vertical"
              className="w-full px-4 space-y-6"
            >
              {/* Section: Purchase Requisition */}
              <div>
                <h3 className="text-base font-semibold mb-2">
                  Purchase Requisition
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                  <Form.Item
                    name="purchase_requisition_id"
                    label="Purchase Requisition"
                    rules={[
                      {
                        required: true,
                        message: "Purchase Requisition is required",
                      },
                    ]}
                  >
                    <Select
                      showSearch
                      placeholder="Select a purchase requisition"
                      optionFilterProp="label"
                      options={(purchaseRequisitions ?? []).map((pr) => ({
                        label: `${pr.product_name} (${pr.purchase_requisition_id})`,
                        value: pr.purchase_requisition_id,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="supplier_id"
                    label="Supplier"
                    rules={[
                      { required: true, message: "Supplier is required" },
                    ]}
                  >
                    <Select
                      showSearch
                      placeholder="Select a supplier"
                      optionFilterProp="label"
                      options={(suppliers ?? []).map((supplier) => ({
                        label: `${supplier.company_name} (${
                          supplier.contact_person || "N/A"
                        })`,
                        value: supplier.supplier_id,
                      }))}
                    />
                  </Form.Item>
                </div>
              </div>

              {/* Section: Order Information */}
              <div>
                <h3 className="text-base font-semibold mb-2">
                  Order Information
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                  <Form.Item name="po_date" label="Purchase Order Date">
                    <DatePicker className="w-full" />
                  </Form.Item>

                  <Form.Item
                    name="expected_delivery_date"
                    label="Expected Delivery Date"
                  >
                    <DatePicker className="w-full" />
                  </Form.Item>

                  <Form.Item name="payment_terms" label="Payment Terms">
                    <Input placeholder="e.g., Next 30 days" />
                  </Form.Item>

                  <Form.Item name="quantity_ordered" label="Quantity Ordered">
                    <Input type="number" placeholder="Enter quantity" />
                  </Form.Item>

                  <Form.Item name="unit_of_measure" label="Unit of Measure">
                    <Input placeholder="e.g., pieces, kg, liters" />
                  </Form.Item>

                  <Form.Item name="price_per_unit" label="Price per Unit">
                    <Input placeholder="Enter price per unit" />
                  </Form.Item>

                  <Form.Item name="total_amount" label="Total Amount">
                    <Input placeholder="Enter total amount" />
                  </Form.Item>
                </div>
              </div>

              {/* Section: Approval & Status */}
              <div>
                <h3 className="text-base font-semibold mb-2">
                  Approval & Status
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                  <Form.Item
                    name="Approve_status"
                    label="Approval Status"
                    rules={[
                      {
                        required: true,
                        message: "Approval Status is required",
                      },
                    ]}
                  >
                    <Select
                      options={(approveStatusOptions ?? []).map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="Status"
                    label="Status"
                    rules={[{ required: true, message: "Status is required" }]}
                  >
                    <Select
                      options={(statusOptions ?? []).map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>
                </div>
              </div>

              {/* Section: Addresses */}
              <div>
                <h3 className="text-base font-semibold mb-2">Addresses</h3>
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                  <Form.Item name="shipping_address" label="Shipping Address">
                    <Input.TextArea
                      rows={2}
                      placeholder="Enter shipping address"
                    />
                  </Form.Item>

                  <Form.Item name="billing_address" label="Billing Address">
                    <Input.TextArea
                      rows={2}
                      placeholder="Enter billing address"
                    />
                  </Form.Item>
                </div>
              </div>

              {/* Section: Remarks & Documents */}
              <div>
                <h3 className="text-base font-semibold mb-2">
                  Additional Details
                </h3>
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                  <Form.Item name="remarks" label="Remarks">
                    <Input.TextArea
                      rows={3}
                      placeholder="Enter any remarks or notes"
                    />
                  </Form.Item>

                  <Form.Item label="Supporting Document">
                    <Upload
                      fileList={supportedFileList}
                      onChange={({ fileList: newFileList }) =>
                        setSupportedFileList(newFileList)
                      }
                      beforeUpload={() => false}
                      maxCount={1}
                    >
                      <Button icon={<FileOutlined />}>Upload Document</Button>
                    </Upload>
                  </Form.Item>
                </div>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Purchase Order Details"
          open={!!viewingPurchaseOrder}
          onCancel={() => setViewingPurchaseOrder(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[600px] top-4"
        >
          {viewingPurchaseOrder && (
            <div className="space-y-2 text-sm sm:text-base">
              <p>
                <strong>PO ID:</strong>{" "}
                <span className="text-blue-600">
                  {viewingPurchaseOrder.purchase_order_id}
                </span>
              </p>
              <p>
                <strong>Requisition ID:</strong>{" "}
                {viewingPurchaseOrder.purchase_requisition_id}
              </p>
              <p>
                <strong>Supplier:</strong>{" "}
                <span className="text-green-600">
                  {getSupplierDisplay(viewingPurchaseOrder.supplier_id)}
                </span>
              </p>
              {/* <p>
                <strong>Product:</strong>{" "}
                {viewingPurchaseOrder.product_id
                  ? getProductDisplay(viewingPurchaseOrder.product_id)
                  : "N/A"}
              </p> */}
              <p>
                <strong>PO Date:</strong>{" "}
                {viewingPurchaseOrder.po_date
                  ? dayjs(viewingPurchaseOrder.po_date).format("YYYY-MM-DD")
                  : "N/A"}
              </p>
              <p>
                <strong>Expected Delivery:</strong>{" "}
                {viewingPurchaseOrder.expected_delivery_date
                  ? dayjs(viewingPurchaseOrder.expected_delivery_date).format(
                      "YYYY-MM-DD"
                    )
                  : "N/A"}
              </p>
              <p>
                <strong>Payment Terms:</strong>{" "}
                {viewingPurchaseOrder.payment_terms || "N/A"}
              </p>
              <p>
                <strong>Quantity Ordered:</strong>{" "}
                {viewingPurchaseOrder.quantity_ordered || "N/A"}{" "}
                {viewingPurchaseOrder.unit_of_measure || ""}
              </p>
              <p>
                <strong>Price per Unit:</strong>{" "}
                {viewingPurchaseOrder.price_per_unit
                  ? `${viewingPurchaseOrder.price_per_unit}`
                  : "N/A"}
              </p>
              <p>
                <strong>Total Amount:</strong>{" "}
                {viewingPurchaseOrder.total_amount
                  ? `${viewingPurchaseOrder.total_amount}`
                  : "N/A"}
              </p>
              <p>
                <strong>Shipping Address:</strong>{" "}
                {viewingPurchaseOrder.shipping_address || "N/A"}
              </p>
              <p>
                <strong>Billing Address:</strong>{" "}
                {viewingPurchaseOrder.billing_address || "N/A"}
              </p>
              <p>
                <strong>Remarks:</strong>{" "}
                {viewingPurchaseOrder.remarks || "N/A"}
              </p>
              <p>
                <strong>Approval Status:</strong>{" "}
                {renderApproveStatusTag(viewingPurchaseOrder.Approve_status)}
              </p>
              <p>
                <strong>Status:</strong>{" "}
                {renderStatusTag(viewingPurchaseOrder.Status)}
              </p>
              <p>
                <strong>Supporting Document:</strong>{" "}
                {viewingPurchaseOrder.supported_file ? (
                  <Button
                    icon={<DownloadOutlined />}
                    onClick={() =>
                      handleDownloadDocument(
                        viewingPurchaseOrder.purchase_order_id!
                      )
                    }
                    size="small"
                    className="ml-2"
                  >
                    Download
                  </Button>
                ) : (
                  "None"
                )}
              </p>
              <p>
                <strong>Created:</strong>{" "}
                {viewingPurchaseOrder.created_at
                  ? dayjs(viewingPurchaseOrder.created_at).format(
                      "YYYY-MM-DD HH:mm"
                    )
                  : "N/A"}
              </p>
              <p>
                <strong>Created By:</strong>{" "}
                {viewingPurchaseOrder.created_by_name || "N/A"} (
                {viewingPurchaseOrder.created_by_id || "N/A"})
              </p>
              <p>
                <strong>Approved By:</strong>{" "}
                {viewingPurchaseOrder.approved_by_name || "N/A"} (
                {viewingPurchaseOrder.approved_by_id || "N/A"})
              </p>
              <p>
                <strong>Approved At:</strong>{" "}
                {viewingPurchaseOrder.approved_at
                  ? dayjs(viewingPurchaseOrder.approved_at).format(
                      "YYYY-MM-DD HH:mm"
                    )
                  : "N/A"}
              </p>
              <p>
                <strong>Updated:</strong>{" "}
                {viewingPurchaseOrder.updated_at
                  ? dayjs(viewingPurchaseOrder.updated_at).format(
                      "YYYY-MM-DD HH:mm"
                    )
                  : "N/A"}
              </p>
            </div>
          )}
        </Modal>

        <ConfirmDialog
          open={confirmDialog.open}
          title={confirmDialog.title}
          message={confirmDialog.message}
          onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
          onConfirm={confirmDialog.onConfirm}
        />
      </div>
    </>
  );
}
