"use client";

import React, { useCallback, useEffect, useMemo, useState } from "react";

import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Row from "antd/es/row";
import "antd/es/row/style";
import Col from "antd/es/col";
import "antd/es/col/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Table from "antd/es/table";
import "antd/es/table/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Collapse from "antd/es/collapse";
import "antd/es/collapse/style";
import Radio from "antd/es/radio";
import "antd/es/radio/style";
import Drawer from "antd/es/drawer";
import "antd/es/drawer/style";
import InputNumber from "antd/es/input-number";
import "antd/es/input-number/style";
import Divider from "antd/es/divider";
import "antd/es/divider/style";

import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SettingOutlined,
  ProfileOutlined,
  ToolOutlined,
  CheckCircleOutlined,
  InboxOutlined,
  DatabaseOutlined,
  FileSearchOutlined,
  BoxPlotOutlined,
  SafetyOutlined,
  TagsOutlined,
  ExperimentOutlined,
  FolderOpenOutlined,
  ContainerOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useSearchParams, useRouter } from "next/navigation";
import {
  createSupplierAssessment,
  deleteSupplierAssessmentsBySupplierId,
  getSupplierAssessmentsBySupplierId,
  updateSupplierAssessment,
  SupplierAssessment,
} from "@/app/api/suppliers/supplierassessment/supplierassessment";

import { getAllSuppliers, Supplier } from "@/app/api/suppliers/suppliers";
import dayjs from "dayjs";

// --- Constants & small helpers ---
const { TextArea } = Input;
const assessmentTypeOptions = ["Initial", "Routine", "Follow Up"] as const;
const statusOptions = ["Pass", "Fail", "Conditional"] as const;
const hazardousOptions = ["Hazardous", "Non_Hazardous"] as const;

// Reusable Yes/No + comment row
const YesNoWithComment: React.FC<{
  name: string;
  label: string;
  commentName?: string;
}> = React.memo(({ name, label, commentName }) => {
  const cmt = commentName ?? `${name}_comment`;
  return (
    <Row gutter={[16, 16]} className="my-2">
      <Col xs={24} md={10}>
        <Form.Item name={name} label={label} rules={[{ required: false }]}>
          <Radio.Group className="flex space-x-4">
            <Radio value="Yes">Yes</Radio>
            <Radio value="No">No</Radio>
          </Radio.Group>
        </Form.Item>
      </Col>
      <Col xs={24} md={14}>
        <Form.Item name={cmt} label="Comments">
          <TextArea rows={3} className="w-full" />
        </Form.Item>
      </Col>
    </Row>
  );
});
YesNoWithComment.displayName = "YesNoWithComment";

// Read-only label/value small component
const L: React.FC<{
  title: string;
  value?: string | number | React.ReactNode;
}> = ({ title, value }) => (
  <div className="flex flex-col mb-3">
    <span className="text-xs font-medium text-gray-500">{title}</span>
    <span className="text-sm text-gray-800 dark:text-gray-200">
      {value ?? "-"}
    </span>
  </div>
);

// All sections (17) — labels taken from your doc
const SECTIONS: {
  key: string;
  title: string;
  icon?: React.ReactNode;
  render: () => React.ReactNode;
}[] = [
  {
    key: "general",
    title: "General Information",
    icon: <HomeOutlined />,
    render: () => (
      <>
        <Row gutter={[16, 16]} className="p-4">
          <Col xs={24} md={8}>
            <Form.Item name="objective" label="Objective">
              <Input className="w-full" />
            </Form.Item>
          </Col>
          <Col xs={24} md={8}>
            <Form.Item
              name="companyName"
              label="Company Name"
              rules={[{ required: true }]}
            >
              <Input className="w-full" readOnly />
            </Form.Item>
          </Col>
          <Col xs={24} md={8}>
            <Form.Item
              name="supplier_type"
              label="Supplier Category - Manufactures / Distributors"
            >
              <Input className="w-full" readOnly />
            </Form.Item>
          </Col>
          <Col xs={24} md={8}>
            <Form.Item name="license_numbers" label="Statutory License details">
              <Input className="w-full" readOnly />
            </Form.Item>
          </Col>
          <Col xs={24} md={16}>
            <Form.Item name="address" label="Address">
              <Input className="w-full" readOnly />
            </Form.Item>
          </Col>
          <Col xs={24} md={6}>
            <Form.Item name="phone_number" label="Phone No.">
              <Input className="w-full" readOnly />
            </Form.Item>
          </Col>
          <Col xs={24} md={6}>
            <Form.Item name="fax" label="Fax">
              <Input className="w-full" readOnly />
            </Form.Item>
          </Col>
          <Col xs={24} md={6}>
            <Form.Item name="email" label="Email">
              <Input className="w-full" readOnly />
            </Form.Item>
          </Col>
          <Col xs={24} md={6}>
            <Form.Item
              name="total_persons_employed"
              label="Total No. of Persons Employed"
            >
              <InputNumber className="w-full" min={0} readOnly />
            </Form.Item>
          </Col>
          <Col xs={24} md={6}>
            <Form.Item name="no_of_shifts" label="No. of Shifts">
              <InputNumber className="w-full" min={0} />
            </Form.Item>
          </Col>
          <Col xs={24} md={6}>
            <Form.Item
              name="no_of_shifts_per_person"
              label="No. of Shifts / Person"
            >
              <InputNumber className="w-full" min={0} />
            </Form.Item>
          </Col>
          <Col xs={24} md={6}>
            <Form.Item name="assessment_date" label="Assement Date">
              <DatePicker className="w-full" format="YYYY-MM-DD" />
            </Form.Item>
          </Col>
          <Col xs={24} md={6}>
            <Form.Item name="assessment_type" label="Assessment Type">
              <Select
                options={assessmentTypeOptions.map((v) => ({
                  label: v,
                  value: v,
                }))}
              />
            </Form.Item>
          </Col>
          <Col xs={24} md={6}>
            <Form.Item name="auditor_name" label="Auditor Name">
              <Input className="w-full" />
            </Form.Item>
          </Col>
          <Col xs={24} md={6}>
            <Form.Item name="date_of_audit" label="Date of Audit">
              <DatePicker className="w-full" format="YYYY-MM-DD" />
            </Form.Item>
          </Col>
        </Row>
      </>
    ),
  },
  {
    key: "management",
    title: "Management",
    icon: <SettingOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <Row gutter={[16, 16]}>
            <Col xs={24} md={8}>
              <Form.Item
                name="key_person_audited"
                label="Name of the key person Audited"
              >
                <Input className="w-full" />
              </Form.Item>
            </Col>
            <Col xs={24} md={8}>
              <Form.Item name="position" label="Position">
                <Input className="w-full" />
              </Form.Item>
            </Col>
            <Col xs={24} md={8}>
              <Form.Item name="signature" label="Signature">
                <Input className="w-full" />
              </Form.Item>
            </Col>

            <Col xs={24} md={12}>
              <Form.Item
                name="type_of_products_manufactured"
                label="Type of Products Manufactured"
              >
                <Input className="w-full" />
              </Form.Item>
            </Col>
            <Col xs={24} md={12}>
              <Form.Item name="product_details" label="Details">
                <TextArea rows={2} />
              </Form.Item>
            </Col>
          </Row>

          <YesNoWithComment
            name="supplier_registration_done"
            label="Is Supplier Registration done?"
          />
          <YesNoWithComment
            name="supplier_register_now_current"
            label="Is Supplier Registration done now current?"
          />
          <Row gutter={[16, 16]}>
            <Col xs={24}>
              <Form.Item
                name="major_customer_details"
                label="Major Customer Details"
              >
                <TextArea rows={3} className="w-full" />
              </Form.Item>
            </Col>
          </Row>
        </div>
      </>
    ),
  },
  {
    key: "infrastructure",
    title: "Infrastructure",
    icon: <ToolOutlined />,
    render: () => (
      <Row gutter={[16, 16]} className="p-4">
        <Col xs={24} md={12}>
          <Form.Item
            name="installed_capacity_details"
            label="Installed capacity Details"
          >
            <TextArea rows={3} className="w-full" />
          </Form.Item>
        </Col>
        <Col xs={24} md={12}>
          <Form.Item
            name="plant_machinery_details"
            label="Plant & Machinery Details"
          >
            <TextArea rows={3} className="w-full" />
          </Form.Item>
        </Col>
      </Row>
    ),
  },
  {
    key: "mgmt_quality",
    title: "Management for Quality",
    icon: <CheckCircleOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="documented_procedure"
            label="Does the company have a documented procedure for their work instructions?"
          />  
          <YesNoWithComment name="follows_gmp" label="Do they follow GMP?" />
          <YesNoWithComment
            name="has_policy_objectives"
            label="Do they have any Policy/Objectives?"
          />
          <YesNoWithComment
            name="manufacturing_process"
            label="Define their manufacturing process"
          />
        </div>
      </>
    ),
  },
  {
    key: "control_spec",
    title: "Control of Specifications",
    icon: <FileSearchOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="system_for_spec_control"
            label="Is there any system followed to control their drawings/specifications?"
          />
          <YesNoWithComment
            name="has_our_specs_test_methods"
            label="Do they have our Specifications and Test methods?"
          />
          <YesNoWithComment
            name="customer_spec_control"
            label="How do they control the Specifications/Drawings provided by the Customers?"
          />
          <YesNoWithComment
            name="customer_property_handling"
            label="Do they have any procedure for customer property handling?"
          />
        </div>
      </>
    ),
  },
  {
    key: "control_procurement",
    title: "Control of Procurement",
    icon: <InboxOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="raw_material_suppliers"
            label="Who are their Suppliers?"
          />
          <YesNoWithComment
            name="raw_materials"
            label="What are their raw materials?"
          />
          <YesNoWithComment
            name="has_supplier_list"
            label="Do they have any suppliers list?"
          />
          <YesNoWithComment
            name="raw_material_rejected"
            label="Is there any raw material rejected for Quality Issues?"
          />
          <YesNoWithComment
            name="corrective_action_for_rejection"
            label="Is there any Corrective action available for the raw material rejections?"
          />
          <YesNoWithComment
            name="quality_checks"
            label="Define their Quality Checks ?"
          />
          <YesNoWithComment
            name="personnel_competent"
            label="Are the personnel competent for carrying out the work?"
          />
          <YesNoWithComment
            name="adequate_testing_resources"
            label="Do they have adequate resource available for the Testing?"
          />
          <YesNoWithComment
            name="measuring_equipment_details"
            label="Details of the measuring Equipment"
          />
          <YesNoWithComment
            name="calibrates_equipment"
            label="Do they calibrate the Equipment?"
          />
          <YesNoWithComment
            name="equipment_status_displayed"
            label="Is the status of the Equipment displayed?"
          />
          <YesNoWithComment
            name="equipment_identified"
            label="Are the Equipment properly identified?"
          />
          <YesNoWithComment
            name="calibration_records_maintained"
            label="Do they maintain records for the Calibration?"
          />
        </div>
      </>
    ),
  },
  {
    key: "batch_control",
    title: "Batch Control",
    icon: <DatabaseOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="batch_numbering_method"
            label="How do they give the batch number for their products?"
          />
          <YesNoWithComment
            name="batch_coding_system"
            label="What is their batch coding system?"
          />
          <YesNoWithComment
            name="has_traceability"
            label="Do they have any adequate traceability?"
          />
          <YesNoWithComment
            name="gives_certificate_of_analysis"
            label="Do they give any certificate of analysis to their customers?"
          />
          <YesNoWithComment
            name="finished_product_label_control"
            label="How do they control the labelling on the Finished Products?"
          />
        </div>
      </>
    ),
  },
  {
    key: "control_process",
    title: "Control of Process",
    icon: <ToolOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="process_control_adequate"
            label="Is it adequate?"
          />
          <YesNoWithComment
            name="documented_inspection_procedure"
            label="Is there any Documented Procedure available for the inspections?"
          />
          <YesNoWithComment
            name="inspection_records_maintained"
            label="Do they maintain records for the inspections?"
          />
          <YesNoWithComment
            name="process_control_details"
            label="How do they control their process?"
          />
        </div>
      </>
    ),
  },
  {
    key: "id_test_status",
    title: "Identification & Test Status",
    icon: <TagsOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="identifies_product_status"
            label="Do they identify the status of the products at each stage of manufacturing?"
          />
          <YesNoWithComment
            name="adequate_to_avoid_mixup"
            label="Is it adequate to control the mix-up?"
          />
        </div>
      </>
    ),
  },
  {
    key: "contamination",
    title: "Contamination Control",
    icon: <ExperimentOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="special_conditions_affect_quality"
            label="Is there any specific or special condition provided by the customer which will have an adverse effect on the quality of the product?"
          />
          <YesNoWithComment
            name="contamination_control_method"
            label="How it is controlled?"
          />
        </div>
      </>
    ),
  },
  {
    key: "control_records",
    title: "Control of Records",
    icon: <FolderOpenOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="records_maintained_each_stage"
            label="Do they maintain any records at each stage of their manufacturing?"
          />
          <YesNoWithComment
            name="record_retention_period"
            label="How long do they retain the records?"
          />
          <YesNoWithComment
            name="records_traceable"
            label="Are they easily traceable?"
          />
          <YesNoWithComment
            name="maintain_test_records"
            label="Do they maintain adequate Inspection Test Record/Files?"
          />
          <YesNoWithComment
            name="inspection_test_records_up_to_date"
            label="Are they up to date?"
          />
        </div>
      </>
    ),
  },
  {
    key: "finished_goods",
    title: "Control of Finished Goods",
    icon: <BoxPlotOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="testing_for_finished_goods"
            label="Do they have any Testing for the Finished Goods?"
          />
          <YesNoWithComment
            name="follows_sampling_plans"
            label="Do they follow any Sampling Plans?"
          />
          <YesNoWithComment
            name="every_batch_tested_as_spec"
            label="Is every batch of Finished Goods tested as per our specification?"
          />
          <YesNoWithComment
            name="retains_finished_goods_samples"
            label="Do they retain any Finished Goods samples (Stock Control Samples)?"
          />
          <YesNoWithComment
            name="retention_years"
            label="For how many years do they retain? (in years)"
          />
        </div>
      </>
    ),
  },
  {
    key: "training",
    title: "Training",
    icon: <ProfileOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="personnel_trained"
            label="Are the personnel trained?"
          />
          <YesNoWithComment
            name="training_effective"
            label="Is the training effective?"
          />
          <YesNoWithComment
            name="training_records_maintained"
            label="Do they maintain the records?"
          />
        </div>
      </>
    ),
  },
  {
    key: "non_conformance",
    title: "Control of Non-Conformance",
    icon: <SafetyOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="identification_of_non_conforming_material"
            label="Is there a proper identification for the non-conforming material?"
          />
          <YesNoWithComment
            name="investigation_for_non_conformance"
            label="Do they carry out any investigation for the non-conforming materials?"
          />
          <YesNoWithComment
            name="disposal_of_non_conformance"
            label="How do they dispose of the non-conforming materials?"
          />
          <YesNoWithComment
            name="non_conformance_records_available"
            label="Is there any record available?"
          />
        </div>
      </>
    ),
  },
  {
    key: "capa",
    title: "Corrective & Preventive Actions",
    icon: <CheckCircleOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="corrective_preventive_actions_taken"
            label="Is any corrective and preventive action taken for rejections?"
          />
          <Row gutter={[16, 16]}>
            <Col xs={24}>
              <YesNoWithComment
                name="handles_customer_complaints"
                label="Do they handle complaints from their customers?"
              />
            </Col>
          </Row>
          <YesNoWithComment
            name="complaint_records_maintained"
            label="Do they maintain any record for the complaints?"
          />
          <YesNoWithComment
            name="corrective_action_effective"
            label="Is the corrective action taken effective?"
          />
          <YesNoWithComment
            name="communicates_corrective_action_to_customers"
            label="How do they communicate the corrective actions to the customers?"
          />
          <YesNoWithComment
            name="feedback_system_from_customers"
            label="Is there any feedback system available from their customers?"
          />
          <YesNoWithComment
            name="process_control_details"
            label="How do they communicate any change in their process or suggestions to the customers?"
          />
          
        </div>
      </>
    ),
  },
  {
    key: "handling_storage",
    title: "Handling & Storage",
    icon: <ContainerOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <Row gutter={[16, 16]}>
            <Col xs={24}>
              <YesNoWithComment
                name="finished_goods_storage_method"
                label="How do they store the Finished Goods?"
              />
            </Col>
          </Row>
          <YesNoWithComment
            name="follows_special_storage_conditions"
            label="Any special storage condition is followed as per the customer’s specification?"
          />
          <YesNoWithComment
            name="storage_condition_records_available"
            label="Is there any record available for this?"
          />
          <YesNoWithComment
            name="assigns_shelf_life"
            label="Do they assign the shelf life for the materials?"
          />
        </div>
      </>
    ),
  },
  {
    key: "packaging_shipping",
    title: "Packaging & Shipping",
    icon: <InboxOutlined />,
    render: () => (
      <>
        <div className="p-4">
          <YesNoWithComment
            name="legislative_requirement_for_shipping"
            label="Is there any legislative requirement required for the products for shipping?"
          />
          <Row gutter={[16, 16]}>
            <Col xs={24} md={12}>
              <YesNoWithComment
                name="mode_of_shipping"
                label="Mode of Shipping"
              />
            </Col>
          </Row>
          <YesNoWithComment
            name="special_packaging_followed"
            label="Do they follow any special packaging for the products to avoid damage during transit or as specified by the customer?"
          />
          <YesNoWithComment
            name="shipping_controls"
            label="What control is followed at the time of shipping?"
          />
          <Row gutter={[16, 16]}>
            <Col xs={24} md={12}>
              <Form.Item
                name="hazardous_or_non_hazardous"
                label="Hazardous/ Non-Hazardous"
                rules={[
                  {
                    required: true,
                    message: "Please select Hazardous / Non_Hazardous",
                  },
                ]}
              >
                <Select
                  options={hazardousOptions.map((v) => ({
                    value: v,
                    label: v,
                  }))}
                  className="w-full"
                />
              </Form.Item>
            </Col>
          </Row>
          <Row gutter={[16, 16]}>
            <Col xs={24}>
              <Form.Item
                name="hazardous_or_non_hazardous_comment"
                label="Hazardous Comment"
              >
                <TextArea rows={3} className="w-full" />
              </Form.Item>
            </Col>
          </Row>
        </div>
      </>
    ),
  },
  {
    key: "summary_status",
    title: "Status & Metadata",
    icon: <TagsOutlined />,
    render: () => (
      <>
        <Row gutter={[16, 16]} className="p-4">
          <Col xs={24} md={8}>
            <Form.Item
              name="assement_status"
              label="Status"
              rules={[{ required: true, message: "Status is required" }]}
            >
              <Select
                options={statusOptions.map((v) => ({ label: v, value: v }))}
                className="w-full"
              />
            </Form.Item>
          </Col>
          <Col xs={24} md={8}>
            <Form.Item
              name="created_by_name"
              label="Supplier Authorized Personnel Name"
            >
              <Input className="w-full" />
            </Form.Item>
          </Col>
          <Col xs={24} md={8}>
            <Form.Item name="updated_by_name" label="Reviewed / Approved By">
              <Input className="w-full" />
            </Form.Item>
          </Col>
        </Row>
      </>
    ),
  },
];

// ---------- Component ----------
export default function SupplierAssessmentPage() {
  const [form] = Form.useForm();
  const [supplier, setSupplier] = useState<Supplier | null>(null);
  const [assessments, setAssessments] = useState<SupplierAssessment[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingAssessment, setEditingAssessment] =
    useState<SupplierAssessment | null>(null);
  const [viewingAssessment, setViewingAssessment] =
    useState<SupplierAssessment | null>(null);
  const [isTableView, setIsTableView] = useState(false);

  const searchParams = useSearchParams();
  const router = useRouter();
  const supplierId = searchParams.get("id");
  const supplierName = searchParams.get("name") || "Unknown Supplier";

  useEffect(() => {
    if (!supplierId) {
      router.push("/suppliers");
      return;
    }

    const fetchAssessments = async () => {
      try {
        // Fetch assessments
        const assessments = await getSupplierAssessmentsBySupplierId(
          supplierId
        );
        setAssessments(assessments);

        // Fetch supplier details
        const suppliers = await getAllSuppliers();
        const selectedSupplier = suppliers.find(
          (s) => s.supplier_id === supplierId
        );
        if (selectedSupplier) {
          setSupplier(selectedSupplier);
        }
      } catch {
        toast.error("Failed to fetch assessments");
      }
    };

    fetchAssessments();
  }, [supplierId, router, form]);

  // Open Add Modal
  const openAddModal = useCallback(() => {
    form.resetFields();
    form.setFieldsValue({
      companyName: supplier?.company_name ?? decodeURIComponent(supplierName),
      supplier_type: supplier?.Suplier_type,
      address: supplier?.address,
      phone_number: supplier?.phone_number,
      fax: supplier?.fax,
      email: supplier?.email,
      license_numbers: supplier?.license_numbers,
      total_persons_employed: supplier?.number_of_employees,
      hazardous_or_non_hazardous: "Non_Hazardous",
      assement_status: "Pass",
      assessment_type: "Initial",
      assessment_date: dayjs(),
      date_of_audit: dayjs(),
    });
    setEditingAssessment(null);
    setIsModalOpen(true);
  }, [form, supplierName, supplier]);

  // Open Edit Modal
  const openEditModal = useCallback(
    (assessment: SupplierAssessment) => {
      form.setFieldsValue({
        ...assessment,
        assessment_date:
          assessment.assessment_date &&
          dayjs(assessment.assessment_date).isValid()
            ? dayjs(dayjs(assessment.assessment_date).format("YYYY-MM-DD"))
            : null,

        date_of_audit:
          assessment.date_of_audit && dayjs(assessment.date_of_audit).isValid()
            ? dayjs(dayjs(assessment.date_of_audit).format("YYYY-MM-DD"))
            : null,

        companyName: supplier?.company_name ?? assessment.companyName,
        address: supplier?.address ?? assessment.address,
        phone_number: supplier?.phone_number ?? assessment.phone_number,
        fax: supplier?.fax ?? assessment.fax,
        email: supplier?.email ?? assessment.email,
        license_numbers:
          supplier?.license_numbers ?? assessment.license_numbers,
        total_persons_employed:
          supplier?.number_of_employees ?? assessment.total_persons_employed,
      });
      setEditingAssessment(assessment);
      setIsModalOpen(true);
    },
    [form, supplier]
  );

  // Handle Delete
  const handleDelete = useCallback(async (supplierId: string) => {
    try {
      await deleteSupplierAssessmentsBySupplierId(supplierId);
      toast.success("Assessment deleted successfully");
      if (supplierId) {
        const updated = await getSupplierAssessmentsBySupplierId(supplierId);
        setAssessments(updated);
      }
    } catch {
      toast.error("Failed to delete supplier assessment");
    }
  }, []);

  // Update handleFormSubmit function
  const handleFormSubmit = useCallback(async () => {
    try {
      const values = await form.validateFields();
      if (!supplierId) {
        toast.error("Supplier ID is required.");
        return;
      }

      // Define backend-supported fields
      const backendFields = [
        "supplier_id",
        "assessment_date",
        "total_persons_employed",
        "no_of_shifts",
        "no_of_shifts_per_person",
        "assessment_type",
        "auditor_name",
        "date_of_audit",
        "objective",
        "key_person_audited",
        "position",
        "signature",
        "type_of_products_manufactured",
        "product_details",
        "supplier_registration_done",
        "supplier_register_now_current",
        "major_customer_details",
        "installed_capacity_details",
        "plant_machinery_details",
        "documented_procedure",
        "documented_procedure_comment",
        "follows_gmp",
        "follows_gmp_comment",
        "has_policy_objectives",
        "has_policy_objectives_comment",
        "manufacturing_process",
        "manufacturing_process_comment",
        "system_for_spec_control",
        "system_for_spec_control_comment",
        "has_our_specs_test_methods",
        "has_our_specs_test_methods_comment",
        "customer_spec_control",
        "customer_spec_control_comment",
        "customer_property_handling",
        "customer_property_handling_comment",
        "raw_materials",
        "raw_materials_comment",
        "raw_material_suppliers",
        "raw_material_suppliers_comment",
        "has_supplier_list",
        "has_supplier_list_comment",
        "raw_material_rejected",
        "raw_material_rejected_comment",
        "corrective_action_for_rejection",
        "corrective_action_for_rejection_comment",
        "quality_checks",
        "quality_checks_comment",
        "personnel_competent",
        "personnel_competent_comment",
        "adequate_testing_resources",
        "adequate_testing_resources_comment",
        "measuring_equipment_details",
        "measuring_equipment_details_comment",
        "calibrates_equipment",
        "calibrates_equipment_comment",
        "equipment_status_displayed",
        "equipment_status_displayed_comment",
        "equipment_identified",
        "equipment_identified_comment",
        "calibration_records_maintained",
        "calibration_records_maintained_comment",
        "batch_numbering_method",
        "batch_numbering_method_comment",
        "batch_coding_system",
        "batch_coding_system_comment",
        "has_traceability",
        "has_traceability_comment",
        "gives_certificate_of_analysis",
        "gives_certificate_of_analysis_comment",
        "finished_product_label_control",
        "finished_product_label_control_comment",
        "process_control_adequate",
        "process_control_adequate_comment",
        "documented_inspection_procedure",
        "documented_inspection_procedure_comment",
        "inspection_records_maintained",
        "inspection_records_maintained_comment",
        "process_control_details",
        "process_control_details_comment",
        "identifies_product_status",
        "identifies_product_status_comment",
        "adequate_to_avoid_mixup",
        "adequate_to_avoid_mixup_comment",
        "special_conditions_affect_quality",
        "special_conditions_affect_quality_comment",
        "contamination_control_method",
        "contamination_control_method_comment",
        "records_maintained_each_stage",
        "records_maintained_each_stage_comment",
        "record_retention_period",
        "record_retention_period_comment",
        "records_traceable",
        "records_traceable_comment",
        "maintain_test_records",
        "maintain_test_records_comment",
        "inspection_test_records_up_to_date",
        "inspection_test_records_up_to_date_comment",
        "testing_for_finished_goods",
        "testing_for_finished_goods_comment",
        "follows_sampling_plans",
        "follows_sampling_plans_comment",
        "every_batch_tested_as_spec",
        "every_batch_tested_as_spec_comment",
        "retains_finished_goods_samples",
        "retains_finished_goods_samples_comment",
        "retention_years",
        "retention_years_comment",
        "personnel_trained",
        "personnel_trained_comment",
        "training_effective",
        "training_effective_comment",
        "training_records_maintained",
        "training_records_maintained_comment",
        "identification_of_non_conforming_material",
        "identification_of_non_conforming_material_comment",
        "investigation_for_non_conformance",
        "investigation_for_non_conformance_comment",
        "disposal_of_non_conformance",
        "disposal_of_non_conformance_comment",
        "non_conformance_records_available",
        "non_conformance_records_available_comment",
        "corrective_preventive_actions_taken",
        "corrective_preventive_actions_taken_comment",
        "handles_customer_complaints",
        "handles_customer_complaints_comment",
        "complaint_records_maintained",
        "complaint_records_maintained_comment",
        "corrective_action_effective",
        "corrective_action_effective_comment",
        "communicates_corrective_action_to_customers",
        "communicates_corrective_action_to_customers_comment",
        "feedback_system_from_customers",
        "feedback_system_from_customers_comment",
        "finished_goods_storage_method",
        "finished_goods_storage_method_comment",
        "follows_special_storage_conditions",
        "follows_special_storage_conditions_comment",
        "storage_condition_records_available",
        "storage_condition_records_available_comment",
        "assigns_shelf_life",
        "assigns_shelf_life_comment",
        "legislative_requirement_for_shipping",
        "legislative_requirement_for_shipping_comment",
        "mode_of_shipping",
        "mode_of_shipping_comment",
        "special_packaging_followed",
        "special_packaging_followed_comment",
        "shipping_controls",
        "shipping_controls_comment",
        "hazardous_or_non_hazardous",
        "hazardous_or_non_hazardous_comment",
        "assement_status",
      ];

      // Filter payload to only include backend-supported fields and handle defaults
      const payload: SupplierAssessment = {
        supplier_id: supplierId,
        companyName: values.companyName ?? supplierName,
        assessment_type: values.assessment_type,
        assement_status: values.assement_status,
        // status: values.assement_status,
        hazardous_or_non_hazardous: values.hazardous_or_non_hazardous,
        auditor_name: values.auditor_name ?? "",
        ...Object.fromEntries(
          Object.entries(values)
            .filter(([key]) => backendFields.includes(key))
            .map(([key, value]) => {
              // Handle fields that expect 'Yes' or 'No' by providing a default if empty
              if (
                [
                  "signature",
                  "documented_procedure",
                  "follows_gmp",
                  "has_policy_objectives",
                  "manufacturing_process",
                  "system_for_spec_control",
                  "has_our_specs_test_methods",
                  "customer_spec_control",
                  "customer_property_handling",
                  "raw_materials",
                  "raw_material_suppliers",
                  "has_supplier_list",
                  "raw_material_rejected",
                  "corrective_action_for_rejection",
                  "quality_checks",
                  "personnel_competent",
                  "adequate_testing_resources",
                  "measuring_equipment_details",
                  "calibrates_equipment",
                  "equipment_status_displayed",
                  "equipment_identified",
                  "calibration_records_maintained",
                  "batch_numbering_method",
                  "batch_coding_system",
                  "has_traceability",
                  "gives_certificate_of_analysis",
                  "finished_product_label_control",
                  "process_control_adequate",
                  "documented_inspection_procedure",
                  "inspection_records_maintained",
                  "process_control_details",
                  "identifies_product_status",
                  "adequate_to_avoid_mixup",
                  "special_conditions_affect_quality",
                  "contamination_control_method",
                  "records_maintained_each_stage",
                  "record_retention_period",
                  "records_traceable",
                  "maintain_test_records",
                  "inspection_test_records_up_to_date",
                  "testing_for_finished_goods",
                  "follows_sampling_plans",
                  "every_batch_tested_as_spec",
                  "retains_finished_goods_samples",
                  "retention_years",
                  "personnel_trained",
                  "training_effective",
                  "training_records_maintained",
                  "identification_of_non_conforming_material",
                  "investigation_for_non_conformance",
                  "disposal_of_non_conformance",
                  "non_conformance_records_available",
                  "corrective_preventive_actions_taken",
                  "handles_customer_complaints",
                  "complaint_records_maintained",
                  "corrective_action_effective",
                  "communicates_corrective_action_to_customers",
                  "feedback_system_from_customers",
                  "finished_goods_storage_method",
                  "follows_special_storage_conditions",
                  "storage_condition_records_available",
                  "assigns_shelf_life",
                  "legislative_requirement_for_shipping",
                  "mode_of_shipping",
                  "special_packaging_followed",
                  "shipping_controls",
                ].includes(key)
              ) {
                return [key, value || "No"];
              }
              return [key, value ?? ""];
            })
        ),
        // 👇 put normalized dates last, so they override raw values
        assessment_date: values.assessment_date
          ? dayjs(values.assessment_date).format("YYYY-MM-DD")
          : "",
        date_of_audit: values.date_of_audit
          ? dayjs(values.date_of_audit).format("YYYY-MM-DD")
          : "",
      };

      if (editingAssessment) {
        // UPDATE
        await updateSupplierAssessment(editingAssessment.supplier_id, payload);
        toast.success("Assessment updated successfully");
      } else {
        // CREATE
        await createSupplierAssessment(payload);
        toast.success("Assessment added successfully");
      }

      setIsModalOpen(false);
      form.resetFields();

      if (supplierId) {
        const updated = await getSupplierAssessmentsBySupplierId(supplierId);
        setAssessments(updated);
      }
    } catch (error) {
      toast.error("Failed to submit assessment");
    }
  }, [form, supplierId, supplierName, editingAssessment]);
  const columns: ColumnsType<SupplierAssessment> = useMemo(
    () => [
      { title: "Supplier ID", dataIndex: "supplier_id", width: 120 },
      {
        title: "Supplier Name",
        dataIndex: "companyName",
        render: (_: string, r) => r.companyName ?? "-",
        width: 200,
      },
      {
        title: "Assessment Date",
        dataIndex: "assessment_date",
        sorter: (a, b) =>
          dayjs(a.assessment_date).unix() - dayjs(b.assessment_date).unix(),
        render: (value: string) =>
          value
            ? `${dayjs(value).format("DD-MM-YYYY")} (${dayjs(value).fromNow()})`
            : "-",
        width: 200,
      },
      {
        title: "Type",
        dataIndex: "assessment_type",
        filters: assessmentTypeOptions.map((t) => ({ text: t, value: t })),
        onFilter: (value, record) => record.assessment_type === value,
        width: 150,
        responsive: ["md"],
      },
      {
        title: "Auditor",
        dataIndex: "auditor_name",
        sorter: (a, b) =>
          (a.auditor_name ?? "").localeCompare(b.auditor_name ?? ""),
        width: 150,
        responsive: ["md"],
      },
      {
        title: "Status",
        dataIndex: "assement_status",
        filters: statusOptions.map((s) => ({ text: s, value: s })),
        onFilter: (value, record) => record.assement_status === value,
        render: (text) => (
          <Tag
            color={
              text === "Pass" ? "green" : text === "Fail" ? "red" : "orange"
            }
          >
            {text}
          </Tag>
        ),
        width: 120,
      },
      {
        title: "Hazardous",
        dataIndex: "hazardous_or_non_hazardous",
        render: (val) => (
          <Tag color={val === "Hazardous" ? "red" : "blue"}>{val}</Tag>
        ),
        width: 150,
      },
      {
        title: "Actions",
        fixed: "right",
        width: 120,
        render: (_, record) => (
          <Space size="small">
            <Tooltip title="View">
              <Button
                icon={<EyeOutlined />}
                onClick={() => setViewingAssessment(record)}
                size="small"
                className="text-blue-600 hover:text-blue-800"
              />
            </Tooltip>
            <Tooltip title="Edit">
              <Button
                icon={<EditOutlined />}
                onClick={() => openEditModal(record)}
                size="small"
                type="primary"
                className="bg-blue-600 hover:bg-blue-700"
              />
            </Tooltip>
            <Tooltip title="Delete">
              <Button
                icon={<DeleteOutlined />}
                onClick={() => handleDelete(record.supplier_id)}
                danger
                size="small"
                className="text-red-600 hover:text-red-800"
              />
            </Tooltip>
          </Space>
        ),
      },
    ],
    [openEditModal, handleDelete]
  );

  // ExpandableRow — collapse with section highlights (concise)
  const ExpandedRow: React.FC<{ rec: SupplierAssessment }> = ({ rec }) => (
    <Collapse
      accordion
      className="bg-gray-50 dark:bg-gray-800 rounded-lg"
      expandIconPosition="right"
    >
      {SECTIONS.map((s) => (
        <Collapse.Panel
          header={
            <span className="flex items-center gap-2 text-sm font-medium">
              {s.icon}
              {s.title}
            </span>
          }
          key={s.key}
          className="p-4"
        >
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {s.key === "general" && (
              <>
                <L title="Objective" value={rec.objective} />
                <L title="Company Name" value={rec.companyName} />
                <L title="Supplier Category" value={rec.supplier_type} />
                <L title="Phone No." value={rec.phone_number} />
                <L title="Email" value={rec.email} />
                <L title="Assessment Type" value={rec.assessment_type} />
                <L title="Auditor Name" value={rec.auditor_name} />
                <L title="Date of Audit" value={rec.date_of_audit} />
              </>
            )}
            {s.key === "mgmt_quality" && (
              <>
                <L
                  title="Documented Procedure"
                  value={rec.documented_procedure}
                />
                <L title="GMP" value={rec.follows_gmp} />
                <L
                  title="Policy/Objectives"
                  value={rec.has_policy_objectives}
                />
                <L
                  title="Manufacturing Process"
                  value={rec.manufacturing_process}
                />
              </>
            )}
            {s.key === "packaging_shipping" && (
              <>
                <L
                  title="Legislative Requirement"
                  value={rec.legislative_requirement_for_shipping}
                />
                <L title="Mode of Shipping" value={rec.mode_of_shipping} />
                <L
                  title="Special Packaging"
                  value={rec.special_packaging_followed}
                />
                <L title="Shipping Controls" value={rec.shipping_controls} />
                <L
                  title="Hazardous/Non-Hazardous"
                  value={rec.hazardous_or_non_hazardous}
                />
              </>
            )}
            {[
              "control_procurement",
              "batch_control",
              "control_records",
              "finished_goods",
            ].includes(s.key) && (
              <L
                title="Quick Overview"
                value="Expand Drawer for full details"
              />
            )}
          </div>
        </Collapse.Panel>
      ))}
    </Collapse>
  );

  const AssessmentCard: React.FC<{ assessment: SupplierAssessment }> = ({
    assessment,
  }) => (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.3 }}
      className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all"
    >
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-3 gap-3">
        <h3 className="text-lg font-semibold text-gray-800 dark:text-white truncate">
          {assessment.assessment_type} Assessment
        </h3>
        <Tag color="purple" className="text-sm">
          {dayjs(assessment.assessment_date).format("YYYY-MM-DD")}
        </Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Auditor:</strong> {assessment.auditor_name ?? "-"}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Status:</strong> {assessment.assement_status}
      </p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1">
        <strong>Hazardous:</strong> {assessment.hazardous_or_non_hazardous}
      </p>
      <div className="mt-3 flex gap-2">
        <Tooltip title="View">
          <Button
            icon={<EyeOutlined />}
            onClick={() => setViewingAssessment(assessment)}
            size="small"
            className="text-blue-600 hover:text-blue-800"
          />
        </Tooltip>
        <Tooltip title="Edit">
          <Button
            icon={<EditOutlined />}
            onClick={() => openEditModal(assessment)}
            size="small"
            type="primary"
            className="bg-blue-600 hover:bg-blue-700"
          />
        </Tooltip>
        <Tooltip title="Delete">
          <Button
            icon={<DeleteOutlined />}
            onClick={() => handleDelete(assessment.supplier_id)}
            danger
            size="small"
            className="text-red-600 hover:text-red-800"
          />
        </Tooltip>
      </div>
    </motion.div>
  );

  return (
    <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
      <motion.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        transition={{ duration: 0.5 }}
      >
        <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          Supplier Assessment - {decodeURIComponent(supplierName)}
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage assessments for the selected supplier.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/suppliers",
            },
            {
              title: "Assessment",
            },
          ]}
        />
        <Button
          icon={<HomeOutlined />}
          onClick={() => router.push("/suppliers")}
          className="mt-3 mb-2 bg-green-600 hover:bg-green-700 text-white"
        >
          Back to Suppliers
        </Button>

        <div className="mt-4">
          <Card title="Assessment List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex justify-end gap-4">
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 rounded-lg"
                >
                  Add Assessment
                </Button>
                <Switch
                  checkedChildren={<TableOutlined />}
                  unCheckedChildren={<AppstoreOutlined />}
                  checked={isTableView}
                  onChange={setIsTableView}
                  className="bg-gray-300"
                />
              </div>
            </div>

            {!isTableView ? (
              <div className="overflow-auto p-4">
                <Table
                  columns={columns}
                  dataSource={assessments}
                  rowKey={(r) =>
                    String(`${r.supplier_id}-${r.assessment_date}`)
                  }
                  scroll={{ x: "max-content" }}
                  pagination={{ pageSize: 10 }}
                  expandable={{
                    expandedRowRender: (record) => <ExpandedRow rec={record} />,
                  }}
                  className="rounded-lg"
                />
              </div>
            ) : (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 xl:grid-cols-4 gap-4 p-4">
                {assessments.map((assessment) => (
                  <AssessmentCard
                    key={String(
                      `${assessment.supplier_id}-${assessment.assessment_date}`
                    )}
                    assessment={assessment}
                  />
                ))}
              </div>
            )}

            {/* Create/Edit Modal */}
            <motion.div
              initial={{ opacity: 0, scale: 0.95 }}
              animate={{ opacity: 1, scale: 1 }}
              exit={{ opacity: 0, scale: 0.95 }}
            >
              <Modal
                title={
                  <span className="text-xl font-semibold">
                    {editingAssessment ? "Edit Assessment" : "Add Assessment"}
                  </span>
                }
                open={isModalOpen}
                onOk={handleFormSubmit}
                onCancel={() => {
                  setIsModalOpen(false);
                  form.resetFields();
                }}
                destroyOnHidden
                width="90vw"
                style={{ top: 20 }}
                styles={{
                  body: { maxHeight: "80vh", overflowY: "auto" },
                  header: { backgroundColor: "#f0f2f5" },
                  footer: { padding: "12px 24px" },
                }}
                footer={null}
                className="rounded-xl"
              >
                <Form form={form} layout="vertical" className="p-4 sm:p-6">
                  <Row gutter={[16, 16]} className="mb-6">
                    <Col xs={24} md={12}>
                      <Form.Item label="Supplier ID">
                        <Input
                          value={supplierId ?? ""}
                          disabled
                          className="bg-gray-100"
                        />
                      </Form.Item>
                    </Col>
                    <Col xs={24} md={12}>
                      <Form.Item
                        name="companyName"
                        label="Company Name"
                        rules={[{ required: true }]}
                      >
                        <Input className="w-full" />
                      </Form.Item>
                    </Col>
                  </Row>

                  <Collapse
                    accordion
                    expandIconPosition="right"
                    className="bg-gray-50 dark:bg-gray-800 rounded-lg"
                  >
                    {SECTIONS.map((s) => (
                      <Collapse.Panel
                        header={
                          <span className="flex items-center gap-2 text-sm font-medium">
                            {s.icon}
                            {s.title}
                          </span>
                        }
                        key={s.key}
                        className="p-4"
                      >
                        {s.render()}
                      </Collapse.Panel>
                    ))}
                  </Collapse>

                  <Divider className="my-6" />
                  <div className="flex justify-end gap-3">
                    <Button
                      onClick={() => {
                        setIsModalOpen(false);
                        form.resetFields();
                      }}
                      className="rounded-lg"
                    >
                      Cancel
                    </Button>
                    <Button
                      type="primary"
                      onClick={handleFormSubmit}
                      className="bg-blue-600 hover:bg-blue-700 rounded-lg"
                    >
                      {editingAssessment ? "Update" : "Create"}
                    </Button>
                  </div>
                </Form>
              </Modal>
            </motion.div>

            {/* VIEWER: Drawer with Tabs (read-only) */}
            <Drawer
              title={
                <span className="text-xl font-semibold">
                  Assessment Details — {viewingAssessment?.companyName ?? ""}
                </span>
              }
              open={!!viewingAssessment}
              onClose={() => setViewingAssessment(null)}
              width={Math.min(
                900,
                typeof window !== "undefined" ? window.innerWidth - 80 : 900
              )}
              className="rounded-xl"
            >
              {viewingAssessment && (
                <Collapse
                  accordion
                  expandIconPosition="right"
                  className="bg-gray-50 dark:bg-gray-800 rounded-lg"
                >
                  {SECTIONS.map((s) => (
                    <Collapse.Panel
                      header={
                        <span className="flex items-center gap-2 text-sm font-medium">
                          {s.icon}
                          {s.title}
                        </span>
                      }
                      key={s.key}
                      className="p-4"
                    >
                      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                        {s.key === "general" && (
                          <>
                            <L
                              title="Objective"
                              value={viewingAssessment.objective}
                            />
                            <L
                              title="Company Name"
                              value={viewingAssessment.companyName}
                            />
                            <L
                              title="Supplier Category"
                              value={viewingAssessment.supplier_type}
                            />
                            <L
                              title="License Details"
                              value={viewingAssessment.license_numbers}
                            />
                            <L
                              title="Address"
                              value={viewingAssessment.address}
                            />
                            <L
                              title="Phone"
                              value={viewingAssessment.phone_number}
                            />
                            <L
                              title="Fax"
                              value={viewingAssessment.fax}
                            />
                            <L title="Email" value={viewingAssessment.email} />
                            <L
                              title="Total Number of Persons Employed"
                              value={viewingAssessment.total_persons_employed}
                            />
                            <L
                              title="Number of Shifts"
                              value={viewingAssessment.no_of_shifts}
                            />
                            <L
                              title="No. of Shifts / Person"
                              value={viewingAssessment.no_of_shifts_per_person}
                            />
                            <L
                              title="Assessment Date"
                              value={viewingAssessment.assessment_date}
                            />
                            <L
                              title="Assessment Type"
                              value={viewingAssessment.assessment_type}
                            />
                            <L
                              title="Auditor Name"
                              value={viewingAssessment.auditor_name}
                            />
                            <L
                              title="Date of Audit"
                              value={viewingAssessment.date_of_audit}
                            />
                          </>
                        )}
                        {s.key === "management" && (
                          <>
                            <L
                              title="Key Person Audited"
                              value={viewingAssessment.key_person_audited}
                            />
                            <L
                              title="Position"
                              value={viewingAssessment.position}
                            />
                            <L
                              title="Signature"
                              value={viewingAssessment.signature}
                            />
                            <L
                              title="Type of Products Manufactured"
                              value={
                                viewingAssessment.type_of_products_manufactured
                              }
                            />
                            <L
                              title="Details"
                              value={viewingAssessment.product_details}
                            />
                            <L
                              title="Supplier Registration Done?"
                              value={
                                viewingAssessment.supplier_registration_done
                              }
                            />
                            <L
                              title="Registration Current?"
                              value={
                                viewingAssessment.supplier_register_now_current
                              }
                            />
                            <L
                              title="Major Customers Details"
                              value={viewingAssessment.major_customer_details}
                            />
                          </>
                        )}
                        {s.key === "infrastructure" && (
                          <>
                            <L
                              title="Installed capacity Details"
                              value={viewingAssessment.installed_capacity_details}
                            />
                            <L
                              title="Plant & Machinery Details"
                              value={viewingAssessment.plant_machinery_details}
                            />
                          </>
                        )}
                        {s.key === "mgmt_quality" && (
                          <>
                            <L
                              title="Does the company have a documented procedure for their work instructions?"
                              value={viewingAssessment.documented_procedure}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.documented_procedure_comment}
                            />
                            <L
                              title="Do they follow GMP?"
                              value={viewingAssessment.follows_gmp}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.follows_gmp_comment}
                            />
                            <L
                              title="Do they have any Policy/Objectives?"
                              value={viewingAssessment.has_policy_objectives}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.has_policy_objectives_comment}
                            />
                            <L
                              title="Define their manufacturing process"
                              value={viewingAssessment.manufacturing_process}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.manufacturing_process_comment}
                            />
                          </>
                        )}
                        {s.key === "control_spec" && (
                          <>
                            <L
                              title="Is there any system followed to control their drawings/specifications?"
                              value={viewingAssessment.system_for_spec_control}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.system_for_spec_control_comment}
                            />
                            <L
                              title="Do they have our Specifications and Test methods?"
                              value={viewingAssessment.has_our_specs_test_methods}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.has_our_specs_test_methods_comment}
                            />
                            <L
                              title="How do they control the Specifications/Drawings provided by the Customers?"
                              value={viewingAssessment.customer_spec_control}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.customer_spec_control_comment}
                            />
                            <L
                              title="Do they have any procedure for customer property handling?"
                              value={viewingAssessment.customer_property_handling}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.customer_property_handling_comment}
                            />
                          </>
                        )}
                        {s.key === "control_procurement" && (
                          <>
                            <L
                              title="Who are their Suppliers?"
                              value={viewingAssessment.raw_material_suppliers}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.raw_material_suppliers_comment}
                            />
                            <L
                              title="What are their raw materials?"
                              value={viewingAssessment.raw_materials}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.raw_materials_comment}
                            />
                            <L
                              title="Do they have any suppliers list?"
                              value={viewingAssessment.has_supplier_list}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.has_supplier_list_comment}
                            />
                            <L
                              title="Is there any raw material rejected for Quality Issues?"
                              value={viewingAssessment.raw_material_rejected}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.raw_material_rejected_comment}
                            />
                            <L
                              title="Is there any Corrective action available for the raw material rejections?"
                              value={
                                viewingAssessment.corrective_action_for_rejection
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.corrective_action_for_rejection_comment
                              }
                            />
                            <L
                              title="Define their Quality Checks ?"
                              value={viewingAssessment.quality_checks}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.quality_checks_comment}
                            />
                            <L
                              title="Are the personnel competent for carrying out the work?"
                              value={viewingAssessment.personnel_competent}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.personnel_competent_comment}
                            />
                            <L
                              title="Do they have adequate resource available for the Testing?"
                              value={
                                viewingAssessment.adequate_testing_resources
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.adequate_testing_resources_comment
                              }
                            />
                            <L
                              title="Details of the measuring Equipment"
                              value={
                                viewingAssessment.measuring_equipment_details
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.measuring_equipment_details_comment
                              }
                            />
                            <L
                              title="Do they calibrate the Equipment?"
                              value={viewingAssessment.calibrates_equipment}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.calibrates_equipment_comment}
                            />
                            <L
                              title="Is the status of the Equipment displayed?"
                              value={
                                viewingAssessment.equipment_status_displayed
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.equipment_status_displayed_comment
                              }
                            />
                            <L
                              title="Are the Equipment properly identified?"
                              value={viewingAssessment.equipment_identified}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.equipment_identified_comment}
                            />
                            <L
                              title="Do they maintain records for the Calibration?"
                              value={
                                viewingAssessment.calibration_records_maintained
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.calibration_records_maintained_comment
                              }
                            />
                          </>
                        )}
                        {s.key === "batch_control" && (
                          <>
                            <L
                              title="How do they give the batch number for their products?"
                              value={viewingAssessment.batch_numbering_method}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.batch_numbering_method_comment}
                            />
                            <L
                              title="What is their batch coding system?"
                              value={viewingAssessment.batch_coding_system}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.batch_coding_system_comment}
                            />
                            <L
                              title="Do they have any adequate traceability?"
                              value={viewingAssessment.has_traceability}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.has_traceability_comment}
                            />
                            <L
                              title="Do they give any certificate of analysis to their customers?"
                              value={
                                viewingAssessment.gives_certificate_of_analysis
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.gives_certificate_of_analysis_comment
                              }
                            />
                            <L
                              title="How do they control the labelling on the Finished Products?"
                              value={
                                viewingAssessment.finished_product_label_control
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.finished_product_label_control_comment
                              }
                            />
                          </>
                        )}
                        {s.key === "control_process" && (
                          <>
                            <L
                              title="Is it adequate?"
                              value={viewingAssessment.process_control_adequate}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.process_control_adequate_comment}
                            />
                            <L
                              title="Is there any Documented Procedure available for the inspections?"
                              value={
                                viewingAssessment.documented_inspection_procedure
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.documented_inspection_procedure_comment
                              }
                            />
                            <L
                              title="Do they maintain records for the inspections?"
                              value={
                                viewingAssessment.inspection_records_maintained
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.inspection_records_maintained_comment
                              }
                            />
                            <L
                              title="How do they control their process?"
                              value={viewingAssessment.process_control_details}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.process_control_details_comment}
                            />
                          </>
                        )}
                        {s.key === "id_test_status" && (
                          <>
                            <L
                              title="Do they identify the status of the products at each stage of manufacturing?"
                              value={
                                viewingAssessment.identifies_product_status
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.identifies_product_status_comment
                              }
                            />
                            <L
                              title="Is it adequate to control the mix-up?"
                              value={viewingAssessment.adequate_to_avoid_mixup}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.adequate_to_avoid_mixup_comment}
                            />
                          </>
                        )}
                        {s.key === "contamination" && (
                          <>
                            <L
                              title="Is there any specific or special condition provided by the customer which will have an adverse effect on the quality of the product?"
                              value={
                                viewingAssessment.special_conditions_affect_quality
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.special_conditions_affect_quality_comment
                              }
                            />
                            <L
                              title="How it is Controlled ?"
                              value={
                                viewingAssessment.contamination_control_method
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.contamination_control_method_comment
                              }
                            />
                          </>
                        )}
                        {s.key === "control_records" && (
                          <>
                            <L
                              title="Do they maintain any records at each stage of their manufacturing?"
                              value={
                                viewingAssessment.records_maintained_each_stage
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.records_maintained_each_stage_comment
                              }
                            />
                            <L
                              title="How long do they retain the records?"
                              value={viewingAssessment.record_retention_period}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.record_retention_period_comment}
                            />
                            <L
                              title="Are they easily traceable?"
                              value={viewingAssessment.records_traceable}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.records_traceable_comment}
                            />
                            <L
                              title="Do they maintain adequate Inspection Test Record/Files?"
                              value={viewingAssessment.maintain_test_records}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.maintain_test_records_comment}
                            />
                            <L
                              title="Are they up to date?"
                              value={
                                viewingAssessment.inspection_test_records_up_to_date
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.inspection_test_records_up_to_date_comment
                              }
                            />
                          </>
                        )}
                        {s.key === "finished_goods" && (
                          <>
                            <L
                              title="Do they have any Testing for the Finished Goods?"
                              value={
                                viewingAssessment.testing_for_finished_goods
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.testing_for_finished_goods_comment
                              }
                            />
                            <L
                              title="Do they follow any Sampling Plans?"
                              value={viewingAssessment.follows_sampling_plans}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.follows_sampling_plans_comment}
                            />
                            <L
                              title="Is every batch of Finished Goods tested as per our specification?"
                              value={
                                viewingAssessment.every_batch_tested_as_spec
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.every_batch_tested_as_spec_comment
                              }
                            />
                            <L
                              title="Do they retain any Finished Goods samples (Stock Control Samples)?"
                              value={
                                viewingAssessment.retains_finished_goods_samples
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.retains_finished_goods_samples_comment
                              }
                            />
                            <L
                              title="For how many years do they retain? (in years)"
                              value={viewingAssessment.retention_years}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.retention_years_comment}
                            />
                          </>
                        )}
                        {s.key === "training" && (
                          <>
                            <L
                              title="Are the personnel trained?"
                              value={viewingAssessment.personnel_trained}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.personnel_trained_comment}
                            />
                            <L
                              title="Is the training effective?"
                              value={viewingAssessment.training_effective}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.training_effective_comment}
                            />
                            <L
                              title="Do they maintain the records?"
                              value={
                                viewingAssessment.training_records_maintained
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.training_records_maintained_comment
                              }
                            />
                          </>
                        )}
                        {s.key === "non_conformance" && (
                          <>
                            <L
                              title="Is there a proper identification for the non-conforming material?"
                              value={
                                viewingAssessment.identification_of_non_conforming_material
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.identification_of_non_conforming_material_comment
                              }
                            />
                            <L
                              title="Do they carry out any investigation for the non-conforming materials?"
                              value={
                                viewingAssessment.investigation_for_non_conformance
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.investigation_for_non_conformance_comment
                              }
                            />
                            <L
                              title="How do they dispose of the non-conforming materials?"
                              value={
                                viewingAssessment.disposal_of_non_conformance
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.disposal_of_non_conformance_comment
                              }
                            />
                            <L
                              title="Is there any record available?"
                              value={
                                viewingAssessment.non_conformance_records_available
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.non_conformance_records_available_comment
                              }
                            />
                          </>
                        )}
                        {s.key === "capa" && (
                          <>
                            <L
                              title="Is any corrective and preventive action taken for rejections?"
                              value={
                                viewingAssessment.corrective_preventive_actions_taken
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.corrective_preventive_actions_taken_comment
                              }
                            />
                            <L
                              title="Do they handle complaints from their customers?"
                              value={
                                viewingAssessment.handles_customer_complaints
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.handles_customer_complaints_comment
                              }
                            />
                            <L
                              title="Do they maintain any record for the complaints?"
                              value={
                                viewingAssessment.complaint_records_maintained
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.complaint_records_maintained_comment
                              }
                            />
                            <L
                              title="Is the corrective action taken effective?"
                              value={
                                viewingAssessment.corrective_action_effective
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.corrective_action_effective_comment
                              }
                            />
                            <L
                              title="How do they communicate the corrective actions to the customers?"
                              value={
                                viewingAssessment.communicates_corrective_action_to_customers
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.communicates_corrective_action_to_customers_comment
                              }
                            />
                            <L
                              title="Is there any feedback system available from their customers?"
                              value={
                                viewingAssessment.feedback_system_from_customers
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.feedback_system_from_customers_comment
                              }
                            />
                            <L
                              title="How do they communicate any change in their process or suggestions to the customers?"
                              value={
                                viewingAssessment.process_control_details
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.process_control_details_comment
                              }
                            />
                          </>
                        )}
                        {s.key === "handling_storage" && (
                          <>
                            <L
                              title="How do they store the Finished Goods?"
                              value={
                                viewingAssessment.finished_goods_storage_method
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.finished_goods_storage_method_comment
                              }
                            />
                            <L
                              title="Any special storage condition is followed as per the customer’s specification?"
                              value={
                                viewingAssessment.follows_special_storage_conditions
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.follows_special_storage_conditions_comment
                              }
                            />
                            <L
                              title="Is there any record available for this?"
                              value={
                                viewingAssessment.storage_condition_records_available
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.storage_condition_records_available_comment
                              }
                            />
                            <L
                              title="Do they assign the shelf life for the materials?"
                              value={viewingAssessment.assigns_shelf_life}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.assigns_shelf_life_comment}
                            />
                          </>
                        )}
                        {s.key === "packaging_shipping" && (
                          <>
                            <L
                              title="Is there any legislative requirement required for the products for shipping?"
                              value={
                                viewingAssessment.legislative_requirement_for_shipping
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.legislative_requirement_for_shipping_comment
                              }
                            />
                            <L
                              title="Mode of Shipping"
                              value={viewingAssessment.mode_of_shipping}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.mode_of_shipping_comment}
                            />
                            
                            <L
                              title="Do they follow any special packaging for the products to avoid damage during transit or as specified by the customer?"
                              value={
                                viewingAssessment.special_packaging_followed
                              }
                            />
                            <L
                              title="Comment"
                              value={
                                viewingAssessment.special_packaging_followed_comment
                              }
                            />
                            <L
                              title="What control is followed at the time of shipping?"
                              value={viewingAssessment.shipping_controls}
                            />
                            <L
                              title="Comment"
                              value={viewingAssessment.shipping_controls_comment}
                            />
                            <L
                              title="Hazardous/Non-Hazardous"
                              value={
                                viewingAssessment.hazardous_or_non_hazardous
                              }
                            />
                            <L
                              title="Hazardous Comment"
                              value={
                                viewingAssessment.hazardous_or_non_hazardous_comment
                              }
                            />
                          </>
                        )}
                        {s.key === "summary_status" && (
                          <>
                            <L
                              title="Overall Status"
                              value={viewingAssessment.assement_status}
                            />
                            <L
                              title="Created By"
                              value={viewingAssessment.created_by_name}
                            />
                            <L
                              title="Updated By"
                              value={
                                (Array.isArray(
                                  viewingAssessment.updated_by_name
                                )
                                  ? viewingAssessment.updated_by_name.join(", ")
                                  : viewingAssessment.updated_by_name) ?? "-"
                              }
                            />
                            <L
                              title="Created At"
                              value={
                                viewingAssessment.created_at
                                  ? dayjs(viewingAssessment.created_at).format(
                                      "YYYY-MM-DD"
                                    )
                                  : "-"
                              }
                            />
                          </>
                        )}
                      </div>
                    </Collapse.Panel>
                  ))}
                </Collapse>
              )}
            </Drawer>
          </Card>
        </div>
      </motion.div>
    </div>
  );
}
