"use client";

import React, { useState, useEffect } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import Tabs from "antd/es/tabs";
import "antd/es/tabs/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Descriptions from "antd/es/descriptions";
import "antd/es/descriptions/style";
import Empty from "antd/es/empty";
import "antd/es/empty/style";
import Table from "antd/es/table";
import "antd/es/table/style";
import Collapse from "antd/es/collapse";
import "antd/es/collapse/style";
import {
  HomeOutlined,
  EyeOutlined,
  EditOutlined,
  DeleteOutlined,
  FileTextOutlined,
  IdcardOutlined,
} from "@ant-design/icons";
import { useSearchParams, useRouter } from "next/navigation";
import { motion } from "framer-motion";
import dayjs from "@/data/utils/dayjs";
import { ColumnsType } from "antd/es/table";
import { ArrowBigLeft } from "lucide-react";
import { getAllSuppliers, Supplier } from "@/app/api/suppliers/suppliers";
import {
  getSupplierAssessmentsBySupplierId,
  SupplierAssessment,
} from "@/app/api/suppliers/supplierassessment/supplierassessment";
import {
  getAllSupplierCertifications,
  SupplierCertification,
} from "@/app/api/suppliers/suppliercertification/suppliercertification/suppliercertification";

const SupplierDetailPage = () => {
  const searchParams = useSearchParams();
  const router = useRouter();
  const supplierId = searchParams.get("id");
  const supplierName = searchParams.get("name") || "Unknown Supplier";

  const [supplier, setSupplier] = useState<Supplier | null>(null);
  const [assessments, setAssessments] = useState<SupplierAssessment[]>([]);
  const [certifications, setCertifications] = useState<SupplierCertification[]>(
    []
  );
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (!supplierId) {
      router.push("/suppliers");
      return;
    }

    const fetchData = async () => {
      try {
        const [suppliers, allCertifications, assessments] = await Promise.all([
          getAllSuppliers(),
          getAllSupplierCertifications(),
          getSupplierAssessmentsBySupplierId(supplierId),
        ]);

        const selectedSupplier = suppliers.find(
          (s) => s.supplier_id === supplierId
        );
        if (!selectedSupplier) throw new Error("Supplier not found");

        setSupplier({
          ...selectedSupplier,
          supplier_id: selectedSupplier.supplier_id ?? "N/A",
          company_name: selectedSupplier.company_name,
          contact_person: selectedSupplier.contact_person ?? "-",
          email: selectedSupplier.email ?? "-",
          phone_number: selectedSupplier.phone_number ?? "-",
          country: selectedSupplier.country ?? "-",
          address: selectedSupplier.address ?? "-",
          company_profile: selectedSupplier.company_profile ?? "-",
          ownership_details: selectedSupplier.ownership_details ?? "-",
          gst_tax_id: selectedSupplier.gst_tax_id ?? "-",
          risk_classification: selectedSupplier.risk_classification,
          status: selectedSupplier.status,
          created_at: selectedSupplier.created_at,
          created_by_name: selectedSupplier.created_by_name || "Unknown",
          updated_by_name: selectedSupplier.updated_by_name || "-",
        });

        const supplierCertifications = allCertifications.filter(
          (c) => c.supplier_id === supplierId
        );
        setCertifications(supplierCertifications);
        setAssessments(assessments);
      } catch (err) {
        console.error("Failed to load supplier details:", err);
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, [supplierId, router]);

  const renderTag = (value: string | number) => {
    const colorMap: { [key: string]: string } = {
      Low: "success",
      Medium: "warning",
      High: "error",
      Active: "success",
      Inactive: "warning",
      BlackListed: "error",
      Pass: "success",
      Fail: "error",
      Conditional: "warning",
      Valid: "success",
      Expired: "error",
      Pending: "warning",
    };

    const iconMap: { [key: string]: React.JSX.Element } = {
      Low: <EyeOutlined />,
      Medium: <EditOutlined />,
      High: <DeleteOutlined />,
      Active: <EyeOutlined />,
      Inactive: <EditOutlined />,
      BlackListed: <DeleteOutlined />,
      Pass: <EyeOutlined />,
      Fail: <DeleteOutlined />,
      Conditional: <EditOutlined />,
      Valid: <EyeOutlined />,
      Expired: <DeleteOutlined />,
      Pending: <EditOutlined />,
    };

    return (
      <Tag
        icon={iconMap[value]}
        color={colorMap[value] || "default"}
        style={{
          whiteSpace: "normal", // ✅ allow wrapping
          wordBreak: "break-word", // ✅ break long words
          maxWidth: "28rem", // ✅ keep it from stretching full width
          display: "inline-block", // ✅ makes wrapping apply
        }}
      >
        {value}
      </Tag>
    );
  };

  const assessmentColumns: ColumnsType<SupplierAssessment> = [
    {
      title: "Date",
      dataIndex: "assessment_date",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    { title: "Type", dataIndex: "assessment_type" },
    { title: "Assessor", dataIndex: "assessor" },
    {
      title: "Score (%)",
      dataIndex: "score",
      render: (text) => `${text || 0}%`,
    },
    { title: "Status", dataIndex: "status", render: renderTag },
    {
      title: "Created At",
      dataIndex: "created_at",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    { title: "Created By", dataIndex: "created_by_name" },
  ];

  const certificationColumns: ColumnsType<SupplierCertification> = [
    { title: "Type", dataIndex: "certification_type" },
    { title: "Number", dataIndex: "certification_number" },
    {
      title: "Issue Date",
      dataIndex: "issue_date",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Expiry Date",
      dataIndex: "expiry_date",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    { title: "Status", dataIndex: "status", render: renderTag },
    {
      title: "Created At",
      dataIndex: "created_at",
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    { title: "Created By", dataIndex: "created_by_name" },
  ];

  if (loading || !supplier)
    return <div className="text-center py-10">Loading...</div>;

  return (
    <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-6 min-h-screen bg-gradient-to-br from-gray-100 to-white dark:from-gray-900 dark:to-gray-800">
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.6, ease: "easeOut" }}
      >
        <Button
          icon={<ArrowBigLeft className="text-green-600" />}
          onClick={() => router.push("/suppliers")}
          className="mb-3 text-green-600 hover:text-green-700 border border-green-600 hover:border-green-700 bg-transparent shadow-none"
        >
          Back to Suppliers
        </Button>
        <h2 className="text-2xl sm:text-3xl md:text-4xl font-extrabold text-gray-900 dark:text-white mb-3 bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
          Supplier Details - {supplier.company_name}
        </h2>
        <p className="text-gray-600 border-b pb-4 dark:text-gray-400 text-base sm:text-lg mb-5">
          Comprehensive overview of {supplier.company_name}.
        </p>
        <Breadcrumb
          className="mb-5 sm:mb-6 text-sm sm:text-base text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            { title: "Suppliers", href: "/suppliers" },
            { title: "Details" },
          ]}
        />
      </motion.div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 mt-3">
        {/* Summary Card */}
        <motion.div
          className="lg:col-span-1 flex flex-col"
          initial={{ scale: 0.95, opacity: 0 }}
          animate={{ scale: 1, opacity: 1 }}
          transition={{ delay: 0.2, duration: 0.5 }}
        >
          <Card className="shadow-2xl bg-white dark:bg-gray-800 rounded-2xl p-6 border-0 transform hover:-translate-y-2 transition-all duration-300 flex-1">
            <h3 className="text-xl font-semibold text-gray-800 dark:text-white mb-4 border-b-2 border-blue-200 pb-2 flex justify-between items-center">
              Summary
              <div className="flex gap-2">
                <Tooltip title="Assessment">
                  <Button
                    icon={<FileTextOutlined />}
                    onClick={() =>
                      router.push(
                        `/suppliers/supplierassessment?id=${supplierId}&name=${encodeURIComponent(
                          supplierName
                        )}`
                      )
                    }
                    size="small"
                    className="bg-purple-500 hover:bg-purple-600 text-white"
                  />
                </Tooltip>
                <Tooltip title="Certification">
                  <Button
                    icon={<IdcardOutlined />}
                    onClick={() =>
                      router.push(
                        `/suppliers/supplierscertification?id=${supplierId}&name=${encodeURIComponent(
                          supplierName
                        )}`
                      )
                    }
                    size="small"
                    className="bg-teal-500 hover:bg-teal-600 text-white"
                  />
                </Tooltip>
              </div>
            </h3>
            <div className="space-y-4">
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Supplier Id:
                </span>
                {renderTag(supplier.supplier_id ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Company Name:
                </span>
                {renderTag(supplier.company_name)}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Contact Person:
                </span>
                {renderTag(supplier.contact_person || "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Supplier Type:
                </span>
                {renderTag(supplier.Suplier_type || "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Email:
                </span>
                {renderTag(supplier.email || "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Phone:
                </span>
                {renderTag(supplier.phone_number || "N/A")}
              </div>
              <div className="flex items-start gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Address:
                </span>
                {renderTag(supplier.address || "N/A")}
              </div>

              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Country:
                </span>
                {renderTag(supplier.country || "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  GST/Tax ID:
                </span>
                {renderTag(supplier.gst_tax_id || "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">Risk:</span>
                {renderTag(supplier.risk_classification)}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Approved Status:
                </span>
                {renderTag(supplier.approved_status)}
              </div>
                            <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Fax:
                </span>
                {renderTag(supplier.fax || "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  License Number:
                </span>
                {renderTag(supplier.license_numbers || "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  critical Info:
                </span>
                {renderTag(supplier.critical_information || "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Employees:
                </span>
                {renderTag(supplier.number_of_employees ?? "N/A")}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Status:
                </span>
                {renderTag(supplier.status)}
              </div>
              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Created:
                </span>
                {renderTag(
                  supplier.created_at
                    ? dayjs(supplier.created_at).format("YYYY-MM-DD")
                    : "N/A"
                )}
              </div>
              <div className="flex items-start gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Company Profile:
                </span>
                {renderTag(supplier.company_profile || "-")}
              </div>

              <div className="flex items-center gap-2">
                <span className="text-gray-700 dark:text-black-300">
                  Ownership:
                </span>
                {renderTag(supplier.ownership_details || "-")}
              </div>
            </div>
          </Card>
        </motion.div>

        {/* Details and Tabs */}
        <motion.div
          className="lg:col-span-2 flex flex-col"
          initial={{ scale: 0.95, opacity: 0 }}
          animate={{ scale: 1, opacity: 1 }}
          transition={{ delay: 0.3, duration: 0.5 }}
        >
          <Card className="shadow-2xl bg-white dark:bg-gray-800 rounded-2xl p-6 border-0 transform hover:-translate-y-2 transition-all duration-300 flex-1">
            <Tabs
              defaultActiveKey="1"
              className="mb-6"
              items={[
                {
                  key: "1",
                  label: "Overview",
                  children: (
                    <div className="space-y-4">
                      <Collapse
                        items={[
                          {
                            key: "1",
                            label: "Basic Information",
                            children: (
                              <Descriptions
                                column={1}
                                size="small"
                                className="text-sm text-gray-700 dark:text-gray-300"
                              >
                                <Descriptions.Item label="Contact">
                                  {supplier.contact_person}
                                </Descriptions.Item>
                                <Descriptions.Item label="Email">
                                  {supplier.email}
                                </Descriptions.Item>
                                <Descriptions.Item label="Phone">
                                  {supplier.phone_number}
                                </Descriptions.Item>
                                <Descriptions.Item label="Country">
                                  {supplier.country}
                                </Descriptions.Item>
                                <Descriptions.Item label="Address">
                                  {supplier.address}
                                </Descriptions.Item>
                                <Descriptions.Item label="GST/Tax ID">
                                  {supplier.gst_tax_id}
                                </Descriptions.Item>
                                <Descriptions.Item label="Created By">
                                  {supplier.created_by_name}
                                </Descriptions.Item>
                              </Descriptions>
                            ),
                          },
                          {
                            key: "2",
                            label: "Company Details",
                            children: (
                              <div className="text-sm text-gray-700 dark:text-gray-300">
                                <p>
                                  <strong>Profile:</strong>{" "}
                                  {supplier.company_profile || "-"}
                                </p>
                                <p>
                                  <strong>Ownership:</strong>{" "}
                                  {supplier.ownership_details || "-"}
                                </p>
                              </div>
                            ),
                          },
                        ]}
                      />
                    </div>
                  ),
                },
                {
                  key: "2",
                  label: "Assessments",
                  children:
                    assessments.length > 0 ? (
                      <Table
                        columns={assessmentColumns}
                        dataSource={assessments}
                        rowKey="id"
                        pagination={{ pageSize: 5 }}
                        scroll={{ x: "max-content" }}
                        className="mb-4 border border-gray-200 dark:border-gray-700 rounded-lg"
                      />
                    ) : (
                      <Empty
                        description="No assessments available"
                        className="py-10"
                      />
                    ),
                },
                {
                  key: "3",
                  label: "Certifications",
                  children:
                    certifications.length > 0 ? (
                      <Table
                        columns={certificationColumns}
                        dataSource={certifications}
                        rowKey="id"
                        pagination={{ pageSize: 5 }}
                        scroll={{ x: "max-content" }}
                        className="mb-4 border border-gray-200 dark:border-gray-700 rounded-lg"
                      />
                    ) : (
                      <Empty
                        description="No certifications available"
                        className="py-10"
                      />
                    ),
                },
              ]}
            />
          </Card>
        </motion.div>
      </div>
    </div>
  );
};

export default SupplierDetailPage;
