"use client";

import React, { useState, useEffect, useCallback } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Table from "antd/es/table";
import "antd/es/table/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Upload from "antd/es/upload";
import "antd/es/upload/style";

import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  DownloadOutlined,
  FileOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import { toast } from "sonner";
import { motion } from "framer-motion";
import { useSearchParams, useRouter } from "next/navigation";
import dayjs from "@/data/utils/dayjs";
import {
  SupplierCertification,
  downloadSupplierCertification,
  getAllSupplierCertifications,
  uploadSupplierCertification,
  updateSupplierCertification,
  deleteSupplierCertification,
} from "@/app/api/suppliers/suppliercertification/suppliercertification/suppliercertification";
import ConfirmDialog from "@/app/components/common/ConfirmDialog";

const { TextArea } = Input;
const certificationTypeOptions = [
  "ISO 9001",
  "ISO 13485",
  "ISO 14001",
  "ISO 45001",
  "CE Marking",
  "FDA Registration",
  "Other",
] as const;
const statusOptions = [
  "Valid",
  "Expired",
  "Pending",
  "Suspended",
  "Withdrawn",
] as const;
const criticalityOptions = ["Low", "Medium", "High", "Critical"] as const;

export default function SupplierCertificationPage() {
  const [form] = Form.useForm();
  const [certifications, setCertifications] = useState<SupplierCertification[]>(
    []
  );
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    title: "",
    message: "",
    onConfirm: () => {},
  });
  const [editingCertification, setEditingCertification] =
    useState<SupplierCertification | null>(null);
  const [viewingCertification, setViewingCertification] =
    useState<SupplierCertification | null>(null);
  const [otherCertification, setOtherCertification] = useState("");
  const [documentFile, setDocumentFile] = useState<File | null>(null);
  const [isGridView, setIsGridView] = useState(false);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const searchParams = useSearchParams();
  const router = useRouter();
  const supplierId = searchParams.get("id");
  const supplierName = searchParams.get("name") || "Unknown Supplier";

  useEffect(() => {
    if (!supplierId) {
      router.push("/suppliers");
      return;
    }

    const fetchCertifications = async () => {
      try {
        const all = await getAllSupplierCertifications();
        const filtered = all.filter((c) => c.supplier_id === supplierId);
        setCertifications(filtered);
      } catch (err) {
        toast.error("Failed to fetch certifications");
      }
    };

    fetchCertifications();
  }, [supplierId, router]);

  const openAddModal = () => {
    form.resetFields();
    setOtherCertification("");
    setDocumentFile(null);
    setEditingCertification(null);
    setIsModalOpen(true);
  };

  const openEditModal = (certification: SupplierCertification) => {
    form.setFieldsValue({
      certificationType: certification.certification_type,
      certificationNumber: certification.certification_number,
      issuingBody: certification.issuing_body,
      scope: certification.scope,
      status: certification.status,
      issueDate: certification.issue_date
        ? dayjs(certification.issue_date)
        : undefined,
      expiryDate: certification.expiry_date
        ? dayjs(certification.expiry_date)
        : undefined,
      accreditationBody: certification.accreditation_body,
      countryOfIssue: certification.country_of_issue,
      renewalDate: certification.renewal_date
        ? dayjs(certification.renewal_date)
        : undefined,
      suspensionDate: certification.suspension_date
        ? dayjs(certification.suspension_date)
        : undefined,
      withdrawalDate: certification.withdrawal_date
        ? dayjs(certification.withdrawal_date)
        : undefined,
      versionNumber: certification.version_number,
      criticality: certification.criticality,
      remarks: certification.remarks,
    });

    if (
      !certificationTypeOptions.includes(
        certification.certification_type as (typeof certificationTypeOptions)[number]
      )
    ) {
      setOtherCertification(certification.certification_type);
      form.setFieldValue("certificationType", "Other");
    } else {
      setOtherCertification("");
    }

    setDocumentFile(null);
    setEditingCertification(certification);
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Certification",
      message: "Are you sure you want to delete this certification?",
      onConfirm: async () => {
        try {
          await deleteSupplierCertification(id);
          toast.success("Certification deleted successfully");
          const all = await getAllSupplierCertifications();
          setCertifications(all.filter((c) => c.supplier_id === supplierId));
        } catch {
          toast.error("Failed to delete certification");
        }
      },
    });
  };
  const handleFormSubmit = async () => {
    try {
      const values = await form.validateFields();

      if (values.certificationType === "Other" && !otherCertification.trim()) {
        toast.error("Please specify the Other Certification Type");
        return;
      }

      const formData = new FormData();
      formData.append("supplierId", supplierId!);
      formData.append("supplierName", supplierName);

      formData.append(
        "certificationType",
        values.certificationType === "Other"
          ? otherCertification
          : values.certificationType
      );
      formData.append("certificationNumber", values.certificationNumber);
      formData.append("issuingBody", values.issuingBody);
      formData.append(
        "issueDate",
        dayjs(values.issueDate).format("YYYY-MM-DD")
      );
      formData.append(
        "expiryDate",
        dayjs(values.expiryDate).format("YYYY-MM-DD")
      );
      formData.append("status", values.status);
      formData.append("scope", values.scope || "");
      formData.append("accreditation_body", values.accreditationBody || "");
      formData.append("country_of_issue", values.countryOfIssue || "");
      formData.append(
        "renewal_date",
        values.renewalDate ? dayjs(values.renewalDate).format("YYYY-MM-DD") : ""
      );
      formData.append(
        "suspension_date",
        values.suspensionDate
          ? dayjs(values.suspensionDate).format("YYYY-MM-DD")
          : ""
      );
      formData.append(
        "withdrawal_date",
        values.withdrawalDate
          ? dayjs(values.withdrawalDate).format("YYYY-MM-DD")
          : ""
      );
      formData.append("version_number", values.versionNumber || "");
      formData.append("criticality", values.criticality || "");
      formData.append("remarks", values.remarks || "");

      if (documentFile) {
        formData.append("document_file", documentFile); // ✅ backend expects this
      }

      if (editingCertification) {
        await updateSupplierCertification(
          editingCertification.document_id || "",
          formData
        );
        toast.success("Certification updated successfully");
      } else {
        await uploadSupplierCertification(formData);
        toast.success("Certification uploaded successfully");
      }

      setIsModalOpen(false);
      form.resetFields();
      setDocumentFile(null);
      setOtherCertification("");

      const all = await getAllSupplierCertifications();
      setCertifications(all.filter((c) => c.supplier_id === supplierId));
    } catch (error) {
      toast.error("Failed to submit certification");
    }
  };
  const certificationType = Form.useWatch("certificationType", form);
  const filteredCertifications = certifications; // Add filtering if needed
  const paginatedCertifications = filteredCertifications.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<SupplierCertification> = [
    {
      title: "Supplier ID",
      dataIndex: "supplier_id",
    },
    {
      title: "Supplier Name",
      dataIndex: "supplier_name",
    },
    {
      title: "Certification Type",
      dataIndex: "certification_type",
      filters: certificationTypeOptions.map((t) => ({ text: t, value: t })),
      onFilter: (value, record) =>
        record.certification_type.includes(value as string),
    },
    {
      title: "Certification Number",
      dataIndex: "certification_number",
      sorter: (a, b) =>
        a.certification_number.localeCompare(b.certification_number),
    },
    {
      title: "Issuing Body",
      dataIndex: "issuing_body",
      sorter: (a, b) => a.issuing_body.localeCompare(b.issuing_body),
    },
    {
      title: "Issue Date",
      dataIndex: "issue_date",
      sorter: (a, b) => dayjs(a.issue_date).unix() - dayjs(b.issue_date).unix(),
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Expiry Date",
      dataIndex: "expiry_date",
      sorter: (a, b) =>
        dayjs(a.expiry_date).unix() - dayjs(b.expiry_date).unix(),
      render: (value: string) =>
        value
          ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
              value
            ).fromNow()})`
          : "-",
    },
    {
      title: "Accreditation Body",
      dataIndex: "accreditation_body",
      sorter: (a, b) =>
        (a.accreditation_body || "").localeCompare(b.accreditation_body || ""),
    },
    {
      title: "Country of Issue",
      dataIndex: "country_of_issue",
      sorter: (a, b) =>
        (a.country_of_issue || "").localeCompare(b.country_of_issue || ""),
    },
    {
      title: "Renewal Date",
      dataIndex: "renewal_date",
      sorter: (a, b) =>
        dayjs(a.renewal_date || "1900-01-01").unix() -
        dayjs(b.renewal_date || "1900-01-01").unix(),
      render: (value: string) =>
        value ? dayjs(value).format("DD-MM-YYYY") : "-",
    },
    {
      title: "Suspension Date",
      dataIndex: "suspension_date",
      sorter: (a, b) =>
        dayjs(a.suspension_date || "1900-01-01").unix() -
        dayjs(b.suspension_date || "1900-01-01").unix(),
      render: (value: string) =>
        value ? dayjs(value).format("DD-MM-YYYY") : "-",
    },
    {
      title: "Withdrawal Date",
      dataIndex: "withdrawal_date",
      sorter: (a, b) =>
        dayjs(a.withdrawal_date || "1900-01-01").unix() -
        dayjs(b.withdrawal_date || "1900-01-01").unix(),
      render: (value: string) =>
        value ? dayjs(value).format("DD-MM-YYYY") : "-",
    },
    {
      title: "Version Number",
      dataIndex: "version_number",
      sorter: (a, b) =>
        (a.version_number || "").localeCompare(b.version_number || ""),
    },
    {
      title: "Criticality",
      dataIndex: "criticality",
      sorter: (a, b) =>
        (a.criticality || "").localeCompare(b.criticality || ""),
    },
    {
      title: "Remarks",
      dataIndex: "remarks",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[150px]">{text || "-"}</span>
        </Tooltip>
      ),
    },
    {
      title: "Status",
      dataIndex: "status",
      filters: statusOptions.map((s) => ({ text: s, value: s })),
      onFilter: (value, record) => record.status === value,
      render: (text) => (
        <Tag
          color={
            text === "Valid" ? "green" : text === "Expired" ? "red" : "orange"
          }
        >
          {text}
        </Tag>
      ),
    },
    {
      title: "Download",
      key: "download",
      render: (_, record) => (
        <Tooltip title="Download">
          <Button
            icon={<DownloadOutlined />}
            onClick={async () => {
              try {
                const blob = await downloadSupplierCertification(
                  record.document_id || ""
                );
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement("a");
                a.href = url;
                a.download = `${record.certification_type}_${record.certification_number}.pdf`;
                a.click();
                window.URL.revokeObjectURL(url);
              } catch {
                toast.error("Download failed");
              }
            }}
            size="small"
            className="text-green-600"
          />
        </Tooltip>
      ),
    },
    {
      title: "Certification Scope",
      dataIndex: "scope",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[150px]">{text || "-"}</span>
        </Tooltip>
      ),
    },
    {
      title: "Created By",
      dataIndex: "created_by_name",
      key: "created_by_name",
      render: (text: string) => text || "-",
    },
    {
      title: "Actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingCertification(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Download">
            <Button
              icon={<DownloadOutlined />}
              onClick={async () => {
                try {
                  const blob = await downloadSupplierCertification(
                    record.document_id || ""
                  );
                  const url = window.URL.createObjectURL(blob);
                  const a = document.createElement("a");
                  a.href = url;
                  a.download = `${record.certification_type}_${record.certification_number}.pdf`;
                  a.click();
                  window.URL.revokeObjectURL(url);
                } catch {
                  toast.error("Download failed");
                }
              }}
              size="small"
              className="text-green-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.document_id || "")}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const CertificationCard = ({
    certification,
  }: {
    certification: SupplierCertification;
  }) => (
    <div className="rounded-2xl bg-white dark:bg-gray-900 shadow-md hover:shadow-lg transition-shadow p-5 w-full">
      {/* Header */}
      <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-3 mb-3">
        <div className="flex flex-col sm:flex-row sm:items-center gap-2">
          <h3 className="text-lg font-semibold text-gray-800 dark:text-white">
            {certification.certification_type}
          </h3>
          <Tag
            color="blue"
            className="text-xs font-medium px-2 py-0.5 rounded-md"
          >
            {dayjs(certification.issue_date).format("DD MMM YYYY")}
          </Tag>
        </div>
        <Tag
          color={certification.status === "Valid" ? "green" : "red"}
          className="text-xs px-2 py-0.5 rounded-md"
        >
          {certification.status}
        </Tag>
      </div>

      {/* Details - Line by line */}
      <div className="flex flex-col gap-2 text-sm text-gray-700 dark:text-gray-300">
        <div>
          <span className="font-medium">Document ID:</span>{" "}
          {certification.document_id || "-"}
        </div>
        <div>
          <span className="font-medium">Supplier ID:</span>{" "}
          {certification.supplier_id}
        </div>
        <div>
          <span className="font-medium">Supplier Name:</span>{" "}
          {certification.supplier_name || "-"}
        </div>
        <div>
          <span className="font-medium">Certification Number:</span>{" "}
          {certification.certification_number}
        </div>
        <div>
          <span className="font-medium">Issued By:</span>{" "}
          {certification.issuing_body}
        </div>
        <div>
          <span className="font-medium">Issue Date:</span>{" "}
          {dayjs(certification.issue_date).format("DD MMM YYYY")}
        </div>
        <div>
          <span className="font-medium">Expiry Date:</span>{" "}
          {dayjs(certification.expiry_date).format("DD MMM YYYY")}
        </div>
        <div>
          <span className="font-medium">Scope:</span> {certification.scope}
        </div>
        <div>
          <span className="font-medium">Accreditation Body:</span>{" "}
          {certification.accreditation_body || "-"}
        </div>
        <div>
          <span className="font-medium">Country of Issue:</span>{" "}
          {certification.country_of_issue || "-"}
        </div>
        <div>
          <span className="font-medium">Renewal Date:</span>{" "}
          {certification.renewal_date
            ? dayjs(certification.renewal_date).format("DD MMM YYYY")
            : "-"}
        </div>
        <div>
          <span className="font-medium">Suspension Date:</span>{" "}
          {certification.suspension_date
            ? dayjs(certification.suspension_date).format("DD MMM YYYY")
            : "-"}
        </div>
        <div>
          <span className="font-medium">Withdrawal Date:</span>{" "}
          {certification.withdrawal_date
            ? dayjs(certification.withdrawal_date).format("DD MMM YYYY")
            : "-"}
        </div>
        <div>
          <span className="font-medium">Version Number:</span>{" "}
          {certification.version_number || "-"}
        </div>
        <div>
          <span className="font-medium">Criticality:</span>{" "}
          {certification.criticality || "-"}
        </div>
        <div>
          <span className="font-medium">Remarks:</span>{" "}
          {certification.remarks || "-"}
        </div>
        <div>
          <span className="font-medium">Created At:</span>{" "}
          {certification.created_at
            ? dayjs(certification.created_at).format("DD MMM YYYY")
            : "-"}
        </div>
        <div>
          <span className="font-medium">Created By (ID):</span>{" "}
          {certification.created_by_id || "-"}
        </div>
        <div>
          <span className="font-medium">Created By (Name):</span>{" "}
          {certification.created_by_name || "-"}
        </div>
        <div>
          <span className="font-medium">File Path:</span>{" "}
          {typeof certification.file_path === "string"
            ? certification.file_path
            : certification.file_path
            ? "File Uploaded"
            : "-"}
        </div>
      </div>

      {/* Actions */}
      <div className="mt-4 flex gap-2 flex-wrap">
        <Tooltip title="View">
          <Button
            icon={<EyeOutlined />}
            onClick={() => setViewingCertification(certification)}
            size="small"
            className="text-blue-600 border-blue-200"
          />
        </Tooltip>
        <Tooltip title="Edit">
          <Button
            icon={<EditOutlined />}
            onClick={() => openEditModal(certification)}
            size="small"
            className="bg-blue-600 hover:bg-blue-700 text-white"
          />
        </Tooltip>
        <Tooltip title="Delete">
          <Button
            icon={<DeleteOutlined />}
            onClick={() => handleDelete(certification.document_id || "")}
            size="small"
            danger
          />
        </Tooltip>
      </div>
    </div>
  );

  return (
    <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
      <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
        Supplier Certification - {decodeURIComponent(supplierName)}
      </h2>
      <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
        Manage certifications for the selected supplier.
      </p>
      <Breadcrumb
        className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
        separator=">"
        items={[
          {
            title: (
              <Tooltip title="Dashboard">
                <HomeOutlined />
              </Tooltip>
            ),
            href: "/dashboard",
          },
          {
            title: "Suppliers",
            href: "/suppliers",
          },
          {
            title: "Certification",
          },
        ]}
      />
      <Button
        icon={<HomeOutlined />}
        onClick={() => router.push("/suppliers")}
        className="mt-3 mb-2 bg-green-600 hover:bg-green-700 text-white"
      >
        Back to Suppliers
      </Button>

      <div className="mt-4">
        <Card title="Certification List" className="p-0 mb-6 shadow-sm">
          <div className="p-4 flex flex-col gap-4">
            <div className="flex justify-end gap-4">
              <Button
                type="primary"
                icon={<PlusOutlined />}
                onClick={openAddModal}
                className="bg-blue-600 hover:bg-blue-700"
              >
                Add Certification
              </Button>
              <Switch
                checkedChildren={<AppstoreOutlined />}
                unCheckedChildren={<TableOutlined />}
                checked={isGridView}
                onChange={setIsGridView}
              />
            </div>
          </div>

          {isGridView ? (
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
              {paginatedCertifications.map((certification) => (
                <CertificationCard
                  key={certification.document_id}
                  certification={certification}
                />
              ))}
            </div>
          ) : (
            <div className="overflow-auto p-4">
              <Table
                columns={columns}
                dataSource={paginatedCertifications}
                rowKey="document_id"
                scroll={{ x: "max-content" }}
                pagination={false}
              />
            </div>
          )}
          <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
            <div className="mb-2 sm:mb-0">
              Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
              <strong>
                {Math.min(
                  currentPage * pageSize,
                  filteredCertifications.length
                )}
              </strong>{" "}
              of <strong>{filteredCertifications.length}</strong> certifications
            </div>
            <Pagination
              current={currentPage}
              pageSize={pageSize}
              total={filteredCertifications.length}
              onChange={(page, size) => {
                setCurrentPage(page);
                setPageSize(size);
              }}
              showSizeChanger
              pageSizeOptions={["10", "20", "50"]}
              size="small"
            />
          </div>
        </Card>
      </div>

      <motion.div
        initial={{ opacity: 0, scale: 0.95 }}
        animate={{ opacity: 1, scale: 1 }}
        exit={{ opacity: 0, scale: 0.95 }}
      >
        <Modal
          title={
            editingCertification ? "Edit Certification" : "Add Certification"
          }
          open={isModalOpen}
          onOk={handleFormSubmit}
          onCancel={() => {
            setIsModalOpen(false);
            form.resetFields();
            setOtherCertification("");
            setDocumentFile(null);
          }}
          destroyOnHidden
          width="95%" // responsive width
          className="!top-5"
          styles={{
            body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line
          }}
        >
          <Form form={form} layout="vertical" className="w-full">
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
              {/* Certification Type */}
              <Form.Item
                name="certificationType"
                label="Certification Type"
                rules={[
                  { required: true, message: "Certification Type is required" },
                ]}
              >
                <Select
                  options={certificationTypeOptions.map((v) => ({
                    label: v,
                    value: v,
                  }))}
                  onChange={(value) => {
                    if (value !== "Other") setOtherCertification("");
                    form.setFieldsValue({ certificationType: value });
                  }}
                />
              </Form.Item>

              {/* Other Certification Type */}
              {certificationType === "Other" && (
                <Form.Item
                  name="otherCertificationType"
                  label="Other Certification Type"
                  rules={[
                    {
                      required: true,
                      message: "Other Certification Type is required",
                    },
                  ]}
                >
                  <Input
                    value={otherCertification}
                    onChange={(e) => setOtherCertification(e.target.value)}
                  />
                </Form.Item>
              )}

              <Form.Item
                name="certificationNumber"
                label="Certification Number"
                rules={[
                  {
                    required: true,
                    message: "Certification Number is required",
                  },
                ]}
              >
                <Input />
              </Form.Item>

              <Form.Item
                name="issuingBody"
                label="Issuing Body"
                rules={[
                  { required: true, message: "Issuing Body is required" },
                ]}
              >
                <Input />
              </Form.Item>

              <Form.Item name="accreditationBody" label="Accreditation Body">
                <Input />
              </Form.Item>

              <Form.Item name="countryOfIssue" label="Country of Issue">
                <Input />
              </Form.Item>

              <Form.Item
                name="issueDate"
                label="Issue Date"
                rules={[{ required: true, message: "Issue Date is required" }]}
              >
                <DatePicker className="w-full" />
              </Form.Item>

              <Form.Item
                name="expiryDate"
                label="Expiry Date"
                rules={[{ required: true, message: "Expiry Date is required" }]}
              >
                <DatePicker className="w-full" />
              </Form.Item>

              <Form.Item name="renewalDate" label="Renewal Date">
                <DatePicker className="w-full" />
              </Form.Item>

              <Form.Item name="suspensionDate" label="Suspension Date">
                <DatePicker className="w-full" />
              </Form.Item>

              <Form.Item name="withdrawalDate" label="Withdrawal Date">
                <DatePicker className="w-full" />
              </Form.Item>

              <Form.Item name="versionNumber" label="Version Number">
                <Input />
              </Form.Item>

              <Form.Item name="criticality" label="Criticality">
                <Select
                  options={criticalityOptions.map((v) => ({
                    label: v,
                    value: v,
                  }))}
                />
              </Form.Item>

              {/* Scope & Remarks */}
              <div className="col-span-full">
                <Form.Item name="scope" label="Scope">
                  <TextArea rows={2} />
                </Form.Item>
              </div>

              <div className="col-span-full">
                <Form.Item name="remarks" label="Remarks">
                  <TextArea rows={2} />
                </Form.Item>
              </div>

              {/* Status */}
              <Form.Item
                name="status"
                label="Status"
                rules={[{ required: true, message: "Status is required" }]}
              >
                <Select
                  options={statusOptions.map((v) => ({ label: v, value: v }))}
                />
              </Form.Item>

              {/* Upload Document */}
              <div className="col-span-full lg:col-span-2">
                <Form.Item
                  label="Upload Document"
                  rules={[
                    {
                      required: !editingCertification,
                      message: "Please upload a document",
                    },
                  ]}
                >
                  <Upload
                    fileList={
                      documentFile
                        ? [
                            {
                              uid: documentFile.name,
                              name: documentFile.name,
                              status: "done",
                              url: "",
                              originFileObj:
                                documentFile as unknown as import("antd/es/upload/interface").RcFile,
                            },
                          ]
                        : []
                    }
                    beforeUpload={() => false}
                    onChange={({ fileList }) =>
                      setDocumentFile(
                        fileList[0]?.originFileObj instanceof File
                          ? (fileList[0].originFileObj as File)
                          : null
                      )
                    }
                    maxCount={1}
                  >
                    <Button icon={<FileOutlined />}>Upload Document</Button>
                  </Upload>
                </Form.Item>
              </div>
            </div>
          </Form>
        </Modal>
      </motion.div>

      <Modal
        title="Certification Details"
        open={!!viewingCertification}
        onCancel={() => setViewingCertification(null)}
        footer={null}
        className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
      >
        {viewingCertification && (
          <div className="space-y-2 text-sm sm:text-base">
            <p>
              <strong>Certification Type:</strong>{" "}
              {viewingCertification.certification_type}
            </p>
            <p>
              <strong>Number:</strong>{" "}
              {viewingCertification.certification_number}
            </p>
            <p>
              <strong>Issuing Body:</strong> {viewingCertification.issuing_body}
            </p>
            <p>
              <strong>Accreditation Body:</strong>{" "}
              {viewingCertification.accreditation_body || "-"}
            </p>
            <p>
              <strong>Country of Issue:</strong>{" "}
              {viewingCertification.country_of_issue || "-"}
            </p>
            <p>
              <strong>Issue Date:</strong>{" "}
              {dayjs(viewingCertification.issue_date).format("YYYY-MM-DD")}
            </p>
            <p>
              <strong>Expiry Date:</strong>{" "}
              {dayjs(viewingCertification.expiry_date).format("YYYY-MM-DD")}
            </p>
            <p>
              <strong>Renewal Date:</strong>{" "}
              {viewingCertification.renewal_date
                ? dayjs(viewingCertification.renewal_date).format("YYYY-MM-DD")
                : "-"}
            </p>
            <p>
              <strong>Suspension Date:</strong>{" "}
              {viewingCertification.suspension_date
                ? dayjs(viewingCertification.suspension_date).format(
                    "YYYY-MM-DD"
                  )
                : "-"}
            </p>
            <p>
              <strong>Withdrawal Date:</strong>{" "}
              {viewingCertification.withdrawal_date
                ? dayjs(viewingCertification.withdrawal_date).format(
                    "YYYY-MM-DD"
                  )
                : "-"}
            </p>
            <p>
              <strong>Version Number:</strong>{" "}
              {viewingCertification.version_number || "-"}
            </p>
            <p>
              <strong>Criticality:</strong>{" "}
              {viewingCertification.criticality || "-"}
            </p>
            <p>
              <strong>Scope:</strong> {viewingCertification.scope || "-"}
            </p>
            <p>
              <strong>Remarks:</strong> {viewingCertification.remarks || "-"}
            </p>
            <p>
              <strong>Status:</strong> {viewingCertification.status}
            </p>
            <p>
              <strong>Created:</strong>{" "}
              {dayjs(viewingCertification.created_at).format("YYYY-MM-DD")}
            </p>
          </div>
        )}
      </Modal>
      <ConfirmDialog
        open={confirmDialog.open}
        title={confirmDialog.title}
        message={confirmDialog.message}
        onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
        onConfirm={confirmDialog.onConfirm}
      />
    </div>
  );
}
