"use client";

import Link from "next/link";
import { useEffect, useState } from "react";
import {
  BoxIcon,
  FileBarChart2Icon,
  BookOpenCheckIcon,
  FileTextIcon,
} from "lucide-react";
import { LineChart, Line, ResponsiveContainer } from "recharts";

interface QmsCardProps {
  title: string;
  description: string;
  href: string;
  count: number;
}

const dummyTrendData = [
  { value: 5 },
  { value: 10 },
  { value: 7 },
  { value: 12 },
  { value: 18 },
  { value: 15 },
  { value: 22 },
];

const getIconByTitle = (title: string) => {
  const key = title.toLowerCase();
  if (key.includes("product")) return <BoxIcon className="w-5 h-5" />;
  if (key.includes("quality")) return <FileBarChart2Icon className="w-5 h-5" />;
  if (key.includes("audit")) return <BookOpenCheckIcon className="w-5 h-5" />;
  if (key.includes("document")) return <FileTextIcon className="w-5 h-5" />;
  return <BoxIcon className="w-5 h-5" />; // Default fallback
};

const QmsCard: React.FC<QmsCardProps> = ({ title, description, href, count }) => {
  const [animatedCount, setAnimatedCount] = useState(0);

  useEffect(() => {
    const duration = 2000;
    const startTime = performance.now();

    const animate = (currentTime: number) => {
      const progress = Math.min((currentTime - startTime) / duration, 1);
      setAnimatedCount(Math.floor(progress * count));

      if (progress < 1) requestAnimationFrame(animate);
    };

    requestAnimationFrame(animate);
  }, [count]);

  const icon = getIconByTitle(title);

  return (
    <div className="bg-gradient-to-br from-white via-gray-50 to-white dark:from-gray-900 dark:to-gray-800 dark:border-gray-700 rounded-2xl shadow-md hover:shadow-xl border border-gray-200 dark:bg-gray-900 transition-all p-6 flex flex-col justify-between hover:scale-[1.02] animate-fade-in-up">
      {/* Header Section */}
      <div className="flex items-start justify-between mb-4">
        <div className="flex flex-col">
          <h3 className="text-base font-medium text-gray-500 dark:text-gray-300">{title}</h3>
          <p className="text-4xl font-bold text-green-600 dark:text-green-400">{animatedCount}</p>
        </div>
        <div className="text-green-600 bg-green-100 dark:bg-green-900/40 dark:text-green-300 rounded-full p-2">
          {icon}
        </div>
      </div>

      {/* Optional Trend Chart */}
      <div className="h-12 mb-4">
        <ResponsiveContainer width="100%" height="100%">
          <LineChart data={dummyTrendData}>
            <Line
              type="monotone"
              dataKey="value"
              stroke="#16a34a"
              strokeWidth={2}
              dot={false}
            />
          </LineChart>
        </ResponsiveContainer>
      </div>

      {/* Description */}
      <p className="text-gray-600 dark:text-gray-400 text-sm mb-4 flex-1">{description}</p>

      {/* Action */}
      <Link href={href}>
        <button className="w-full mt-auto bg-green-600 hover:bg-green-700 text-white py-2 px-4 rounded-lg text-sm font-semibold transition-all">
          View {title}
        </button>
      </Link>
    </div>
  );
};

export default QmsCard;
