'use client';

import Image from 'next/image';
import { useState } from 'react';
import { useRouter } from 'next/navigation';
import { useAuth } from '@/context/AuthContext';
import { useForm } from 'react-hook-form';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { Eye, EyeOff, Lock, User } from 'lucide-react';
import { toast } from 'sonner';
import Link from 'next/link';
import AnimatedLoginSpinner from '../components/common/AnimatedSpinner';

const loginSchema = z.object({
  username: z.string().min(1, 'Username is required'),
  password: z.string().min(1, 'Password is required'),
  remember: z.boolean().optional(),
});

type LoginData = z.infer<typeof loginSchema>;

export default function LoginPage() {
  const router = useRouter();
  const { login } = useAuth();
  const [loading, setLoading] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [showAnimatedSpinner, setShowAnimatedSpinner] = useState(false);
  const {
    register,
    handleSubmit,
    formState: { errors },
  } = useForm<LoginData>({
    resolver: zodResolver(loginSchema),
    defaultValues: {
      username: '',
      password: '',
      remember: false,
    },
  });

 const onSubmit = async (data: LoginData) => {
  setLoading(true);
  const success = await login(data.username, data.password, data.remember || false);

  setLoading(false);

  if (success) {
     setShowAnimatedSpinner(true); 
    if (data.remember) {
      localStorage.setItem('remembered', 'true');
    }
    router.push('/dashboard');
    toast.success('Login successfully!');
  } else {
    toast.error('Invalid credentials');
  }
};

  return (
    <>
     {showAnimatedSpinner && <AnimatedLoginSpinner />}
    <div className="min-h-screen flex flex-col md:flex-row bg-gradient-to-r from-green-100 to-white">
      {/* Left full-height image with overlay content */}
      <div className="hidden md:block relative w-full md:w-1/2 h-screen animate-fade-in-left">
        <Image
          src="/login.jpg"
          alt="Illustration"
          fill
          className="object-cover"
          priority
        />
      </div>

      {/* Right login form */}
      <div className="w-full md:w-1/2 flex items-center justify-center px-4 py-10 min-h-screen md:min-h-0">
        <div className="w-full max-w-md bg-white px-8 py-10 rounded-2xl shadow-xl border border-gray-200 animate-fade-in-up">
          <Image
            src="/image.png"
            alt="IATT Logo"
            width={120}
            height={40}
            priority
            className="mx-auto mb-6"
          />
          <h2 className="text-xl font-bold mb-6 text-center">Login to QMS</h2>

          <form onSubmit={handleSubmit(onSubmit)} className="space-y-5">
            {/* Username */}
            <div>
              <label htmlFor="username" className="block text-sm font-medium text-gray-700 mb-1">
                Username
              </label>
              <div className="relative">
                <span className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400">
                  <User className="w-4 h-4" />
                </span>
                <input
                  id="username"
                  placeholder="Enter your username"
                  autoFocus
                  disabled={loading}
                  {...register('username')}
                  type="text"
                  className="w-full pl-10 p-2 bg-gray-50 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500"
                />
              </div>
              {errors.username && (
                <p className="text-red-500 text-sm mt-1">{errors.username.message}</p>
              )}
            </div>

            {/* Password */}
            <div>
              <label htmlFor="password" className="block text-sm font-medium text-gray-700 mb-1">
                Password
              </label>
              <div className="relative">
                {/* Left lock icon */}
                <span className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400">
                  <Lock className="w-4 h-4" />
                </span>

                {/* Password field */}
                <input
                  id="password"
                  placeholder="Enter your password"
                  disabled={loading}
                  {...register('password')}
                  type={showPassword ? 'text' : 'password'}
                  className="w-full p-2 pl-10 pr-10 bg-gray-50 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500"
                />

                {/* Eye toggle */}
                <button
                  type="button"
                  onClick={() => setShowPassword(!showPassword)}
                  className="absolute inset-y-0 right-3 flex items-center text-green-600"
                >
                  {showPassword ? <EyeOff size={18} /> : <Eye size={18} />}
                </button>
              </div>

              {errors.password && (
                <p className="text-red-500 text-sm mt-1">{errors.password.message}</p>
              )}
            </div>

            {/* Remember & Forgot */}
            <div className="flex items-center justify-between text-sm">
              <label className="flex items-center gap-2">
                <input
                  type="checkbox"
                  disabled={loading}
                  {...register('remember')}
                  className="accent-green-600"
                />
                Remember me
              </label>
              <Link href="/login/forgotpassword" className="text-green-600 hover:underline text-sm">
                Forgot password?
              </Link>
            </div>

            {/* Submit */}
            <button
              type="submit"
              disabled={loading}
              className="w-full bg-green-600 text-white py-2 rounded-lg hover:bg-green-700 flex items-center justify-center disabled:opacity-50"
            >
              {loading && (
                <span className="animate-spin border-2 border-white border-t-transparent rounded-full h-4 w-4 mr-2"></span>
              )}
              {loading ? 'Logging in...' : 'Login'}
            </button>

            {/* Footer links */}
            <div className="text-xs text-center text-gray-400 mt-4">
              <a href="#" className="hover:underline">Privacy Policy</a> · <a href="#" className="hover:underline">Terms of Service</a>
            </div>
          </form>
        </div>
      </div>
    </div>
    </>
  );

}
