"use client";

import React, { useCallback, useState } from "react";
import {
  Button,
  Card,
  Col,
  DatePicker,
  Form,
  Input,
  Modal,
  Row,
  Select,
  Space,
  Switch,
  Tooltip,
  Tag,
  Breadcrumb,
  Pagination,
  Dropdown,
} from "antd";
import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  AppstoreOutlined,
  TableOutlined,
  CheckCircleTwoTone,
  StopOutlined,
  WarningOutlined,
  SearchOutlined,
  DownloadOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  MinusCircleOutlined,
  CloseCircleOutlined,
  InfoCircleOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import dayjs from "dayjs";
import DataTable from "../components/common/DataTable";
import WidgetCard from "../components/common/WidgetCard";
import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import { SupplierVisit } from "../types/SupplierVisit";
import { toast } from "sonner";
import { motion } from "framer-motion";

const { RangePicker } = DatePicker;
const visitTypes = ["Initial", "Follow-up", "Routine"] as const;
const statusOptions = ["Active", "Inactive", "BlackListed"] as const;
const riskOptions = ["Low", "Medium", "High"] as const;

export default function SupplierVisitPage() {
  const [form] = Form.useForm();
  const [visits, setVisits] = useState<SupplierVisit[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingVisit, setEditingVisit] = useState<SupplierVisit | null>(null);
  const [viewingVisit, setViewingVisit] = useState<SupplierVisit | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    visitType: undefined,
    riskClassification: undefined,
    status: undefined,
  });
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();

  const openAddModal = () => {
    form.resetFields();
    setEditingVisit(null);
    setIsModalOpen(true);
  };

  const openEditModal = (visit: SupplierVisit) => {
    form.setFieldsValue({
      ...visit,
      scheduledMonth: visit.scheduledMonth ? dayjs(visit.scheduledMonth, "YYYY-MM") : undefined,
      completionDate: visit.completionDate ? dayjs(visit.completionDate, "YYYY-MM-DD") : undefined,
    });
    setEditingVisit(visit);
    setIsModalOpen(true);
  };

  const handleDelete = (id: string) => {
    Modal.confirm({
      title: "Are you sure you want to delete this visit?",
      onOk: () => {
        setVisits((prev) => prev.filter((v) => v.id !== id));
        toast.success("Visit deleted successfully");
      },
      onCancel: () => {},
    });
  };

  const handleBulkDelete = () => {
    Modal.confirm({
      title: `Are you sure you want to delete ${selectedRows.length} visits?`,
      onOk: () => {
        setVisits((prev) => prev.filter((v) => !selectedRows.includes(v.id.toString())));
        setSelectedRows([]);
        toast.success("Visits deleted successfully");
      },
    });
  };

  const handleFormSubmit = () => {
    form.validateFields().then((values) => {
      const data: SupplierVisit = {
        ...values,
        id: editingVisit ? editingVisit.id : Date.now().toString(),
        scheduledMonth: dayjs(values.scheduledMonth).format("YYYY-MM"),
        completionDate: values.completionDate ? dayjs(values.completionDate).format("YYYY-MM-DD") : undefined,
        created_at: editingVisit ? editingVisit.created_at : dayjs().toISOString(),
      };
      if (editingVisit) {
        setVisits((prev) => prev.map((v) => (v.id === editingVisit.id ? data : v)));
        toast.success("Visit updated successfully");
      } else {
        setVisits((prev) => [...prev, data]);
        toast.success("Visit added successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
    });
  };

  const renderTag = (value: string) => {
    const colorMap: Record<string, string> = {
      Initial: "blue",
      "Follow-up": "cyan",
      Routine: "purple",
      Low: "green",
      Medium: "orange",
      High: "red",
      Active: "green",
      Inactive: "gold",
      BlackListed: "red",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Initial: <InfoCircleOutlined />,
      "Follow-up": <CheckCircleOutlined />,
      Routine: <CheckCircleOutlined />,
      Low: <CheckCircleOutlined />,
      Medium: <PauseCircleOutlined />,
      High: <FireOutlined />,
      Active: <CheckCircleOutlined />,
      Inactive: <MinusCircleOutlined />,
      BlackListed: <CloseCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  };

  const filteredVisits = visits.filter((visit) => {
    const matchesSearch =
      visit.supplier.toLowerCase().includes(searchTerm.toLowerCase()) ||
      visit.place.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDate =
      !dateRange ||
      (visit.created_at &&
        (!dateRange[0] ||
          dayjs(visit.created_at).isSame(dateRange[0]) ||
          dayjs(visit.created_at).isAfter(dateRange[0])) &&
        (!dateRange[1] ||
          dayjs(visit.created_at).isSame(dateRange[1]) ||
          dayjs(visit.created_at).isBefore(dateRange[1])));
    const matchesAdvanced =
      (!advancedFilters.visitType || visit.visitType === advancedFilters.visitType) &&
      (!advancedFilters.riskClassification || visit.riskClassification === advancedFilters.riskClassification) &&
      (!advancedFilters.status || visit.status === advancedFilters.status);
    return matchesSearch && matchesDate && matchesAdvanced;
  });

  const paginatedVisits = filteredVisits.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );

  const columns: ColumnsType<SupplierVisit> = [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(checked ? filteredVisits.map((v) => v.id.toString()) : []);
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(record.id.toString())}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, record.id.toString()]
                : prev.filter((id) => id !== record.id.toString())
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Supplier",
      dataIndex: "supplier",
      key: "supplier",
      sorter: (a, b) => a.supplier.localeCompare(b.supplier),
      render: (text) => <span className="text-green-600 font-medium">{text}</span>,
      filterSearch: true,
      onFilter: (value, record) =>
        record.supplier.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(visits.map((v) => v.supplier))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Visit Type",
      dataIndex: "visitType",
      key: "visitType",
      sorter: (a, b) => a.visitType.localeCompare(b.visitType),
      render: renderTag,
      filterSearch: true,
      onFilter: (value, record) => record.visitType === value,
      filters: visitTypes.map((type) => ({ text: type, value: type })),
    },
    {
      title: "Place",
      dataIndex: "place",
      key: "place",
      sorter: (a, b) => a.place.localeCompare(b.place),
      filterSearch: true,
      onFilter: (value, record) =>
        record.place.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(visits.map((v) => v.place))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Material",
      dataIndex: "material",
      key: "material",
      sorter: (a, b) => a.material.localeCompare(b.material),
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text}</span>
        </Tooltip>
      ),
    },
    {
      title: "Scheduled Month",
      dataIndex: "scheduledMonth",
      key: "scheduledMonth",
      sorter: (a, b) => a.scheduledMonth.localeCompare(b.scheduledMonth),
      render: (text) => dayjs(text).format("YYYY-MM"),
    },
    {
      title: "Auditor Name",
      dataIndex: "auditorName",
      key: "auditorName",
      sorter: (a, b) => a.auditorName.localeCompare(b.auditorName),
      filterSearch: true,
      onFilter: (value, record) =>
        record.auditorName.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(visits.map((v) => v.auditorName))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Completion Date",
      dataIndex: "completionDate",
      key: "completionDate",
      sorter: (a, b) =>
        new Date(a.completionDate ?? 0).getTime() - new Date(b.completionDate ?? 0).getTime(),
      render: (text) => (text ? dayjs(text).format("YYYY-MM-DD") : "-"),
    },
    {
      title: "Findings",
      dataIndex: "findings",
      key: "findings",
      render: (text) => (
        <Tooltip title={text}>
          <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">{text || "-"}</span>
        </Tooltip>
      ),
    },
    {
      title: "Risk",
      dataIndex: "riskClassification",
      key: "riskClassification",
      filters: riskOptions.map((r) => ({ text: r, value: r })),
      onFilter: (value, record) => record.riskClassification === value,
      render: renderTag,
      sorter: (a, b) => a.riskClassification.localeCompare(b.riskClassification),
    },
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      filters: statusOptions.map((s) => ({ text: s, value: s })),
      onFilter: (value, record) => record.status === value,
      render: renderTag,
      sorter: (a, b) => a.status.localeCompare(b.status),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingVisit(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.id)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ];

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "supplier", label: "Supplier" },
    { key: "visitType", label: "Visit Type" },
    { key: "place", label: "Place" },
    { key: "material", label: "Material" },
    { key: "scheduledMonth", label: "Scheduled Month" },
    { key: "auditorName", label: "Auditor Name" },
    { key: "completionDate", label: "Completion Date" },
    { key: "findings", label: "Findings" },
    { key: "riskClassification", label: "Risk" },
    { key: "status", label: "Status" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] = useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = columns.filter(
    (col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)
  );

  const VisitCard = ({ visit }: { visit: SupplierVisit }) => (
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">{visit.supplier}</h3>
        <Tag color="blue">{visit.visitType}</Tag>
      </div>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Place:</strong> {visit.place}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Risk:</strong> {renderTag(visit.riskClassification)}</p>
      <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Status:</strong> {renderTag(visit.status)}</p>
      <p className="text-sm text-gray-500 dark:text-gray-400 text-xs"><strong>Scheduled:</strong> {dayjs(visit.scheduledMonth).format("YYYY-MM")}</p>
      <div className="mt-2 flex gap-2">
        <Tooltip title="View">
          <Button icon={<EyeOutlined />} onClick={() => setViewingVisit(visit)} size="small" className="text-blue-600" />
        </Tooltip>
        <Tooltip title="Edit">
          <Button icon={<EditOutlined />} onClick={() => openEditModal(visit)} size="small" type="primary" className="bg-blue-600 hover:bg-blue-700" />
        </Tooltip>
        <Tooltip title="Delete">
          <Button icon={<DeleteOutlined />} onClick={() => handleDelete(visit.id)} size="small" danger className="text-red-600" />
        </Tooltip>
      </div>
    </div>
  );

  const total = visits.length;
  const active = visits.filter((v) => v.status === "Active").length;
  const blackListed = visits.filter((v) => v.status === "BlackListed").length;
  const highRisk = visits.filter((v) => v.riskClassification === "High").length;

  // Sample trend data for widgets
  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-4 md:px-6 lg:px-8 py-4">
        <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          Supplier Visits <span className="text-gray-500 text-base sm:text-lg font-medium">({visits.length})</span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Manage supplier visit schedules and findings.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Supplier Visits",
            },
          ]}
        />

        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Visits"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All scheduled supplier visits"
            trendData={trendData}
          />
          <WidgetCard
            title="Active Visits"
            value={active}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Currently active visits"
            trendData={trendData}
          />
          <WidgetCard
            title="BlackListed"
            value={blackListed}
            icon={<StopOutlined />}
            color="red"
            percentage={-3}
            description="Visits with blacklisted status"
            trendData={trendData}
          />
          <WidgetCard
            title="High Risk"
            value={highRisk}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="High-risk visits"
            trendData={trendData}
          />
        </div>

        <div className="mt-4">
          <Card title="Supplier Visit List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier or Place"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Visit
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Visit Type"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, visitType: value }))
                    }
                    allowClear
                    options={visitTypes.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Risk Level"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, riskClassification: value }))
                    }
                    allowClear
                    options={riskOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, status: value }))
                    }
                    allowClear
                    options={statusOptions.map((v) => ({ label: v, value: v }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="visit_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.visitType || advancedFilters.riskClassification || advancedFilters.status) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.visitType && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          visitType: undefined,
                        }))
                      }
                    >
                      Visit Type: {advancedFilters.visitType}
                    </Tag>
                  )}
                  {advancedFilters.riskClassification && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          riskClassification: undefined,
                        }))
                      }
                    >
                      Risk Level: {advancedFilters.riskClassification}
                    </Tag>
                  )}
                  {advancedFilters.status && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          status: undefined,
                        }))
                      }
                    >
                      Status: {advancedFilters.status}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
                {paginatedVisits.map((visit) => (
                  <VisitCard key={visit.id} visit={visit} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<SupplierVisit>
                  columns={filteredColumns}
                  dataSource={paginatedVisits}
                  rowKey="id"
                  scrollX="max-content"
                  className="mt-4"
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>{Math.min(currentPage * pageSize, filteredVisits.length)}</strong> of{" "}
                <strong>{filteredVisits.length}</strong> visits
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredVisits.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={editingVisit ? "Edit Supplier Visit" : "Add Supplier Visit"}
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            destroyOnHidden
            className="w-full max-w-[90vw] sm:max-w-[600px] md:max-w-[800px] lg:max-w-[1000px] top-4"
          >
            <Form form={form} layout="vertical" className="px-2 sm:px-4">
              <Row gutter={[16, 16]}>
                <Col xs={24} sm={12}>
                  <Form.Item name="supplier" label="Supplier" rules={[{ required: true, message: "Supplier is required" }]}>
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="visitType" label="Visit Type" rules={[{ required: true, message: "Visit Type is required" }]}>
                    <Select options={visitTypes.map((v) => ({ label: v, value: v }))} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="place" label="Place">
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="material" label="Material">
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="scheduledMonth" label="Scheduled Month" rules={[{ required: true, message: "Scheduled Month is required" }]}>
                    <DatePicker picker="month" className="w-full" />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="auditorName" label="Auditor Name">
                    <Input />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="completionDate" label="Completion Date">
                    <DatePicker className="w-full" />
                  </Form.Item>
                </Col>
                <Col xs={24}>
                  <Form.Item name="findings" label="Findings / Description">
                    <Input.TextArea rows={3} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="riskClassification" label="Risk Classification" rules={[{ required: true, message: "Risk Classification is required" }]}>
                    <Select options={riskOptions.map((v) => ({ label: v, value: v }))} />
                  </Form.Item>
                </Col>
                <Col xs={24} sm={12}>
                  <Form.Item name="status" label="Status" rules={[{ required: true, message: "Status is required" }]}>
                    <Select options={statusOptions.map((v) => ({ label: v, value: v }))} />
                  </Form.Item>
                </Col>
              </Row>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Supplier Visit Details"
          open={!!viewingVisit}
          onCancel={() => setViewingVisit(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
        >
          {viewingVisit && (
            <div className="space-y-2 text-sm sm:text-base">
              <p><strong>Supplier:</strong> <span className="text-green-600">{viewingVisit.supplier}</span></p>
              <p><strong>Visit Type:</strong> {viewingVisit.visitType}</p>
              <p><strong>Place:</strong> {viewingVisit.place || "-"}</p>
              <p><strong>Material:</strong> {viewingVisit.material || "-"}</p>
              <p><strong>Scheduled Month:</strong> {dayjs(viewingVisit.scheduledMonth).format("YYYY-MM")}</p>
              <p><strong>Auditor Name:</strong> {viewingVisit.auditorName || "-"}</p>
              <p><strong>Completion Date:</strong> {viewingVisit.completionDate ? dayjs(viewingVisit.completionDate).format("YYYY-MM-DD") : "-"}</p>
              <p><strong>Findings:</strong> {viewingVisit.findings || "-"}</p>
              <p><strong>Risk Classification:</strong> {viewingVisit.riskClassification}</p>
              <p><strong>Status:</strong> {viewingVisit.status}</p>
              <p><strong>Created:</strong> {dayjs(viewingVisit.created_at).format("YYYY-MM-DD")}</p>
            </div>
          )}
        </Modal>
      </div>
    </>
  );
}