"use client";

import { createContext, useContext, useState, ReactNode } from 'react';


// Define the User interface to match the API response
interface User {
  user_id: string;
  name: string;
  role: string;
}

// Define the expected API response structure
interface LoginResponse {
  user_id: string;
  name: string;
  role: string;
  access_token: string;
  token_type: string;
}

interface AuthContextType {
  user: User | null;
  login: (username: string, password: string, remember_me: boolean) => Promise<boolean>;
  logout: () => void;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

export const AuthProvider = ({ children }: { children: ReactNode }) => {
  const [user, setUser] = useState<User | null>(null);

  const login = async (username: string, password: string, remember_me: boolean): Promise<boolean> => {
    try {
      const response = await fetch('/api/login', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          username,
          password,
          remember_me: remember_me ? 'true' : 'false',
        }),
      });

      if (!response.ok) {
        const errData = await response.json();
        throw new Error(errData.detail || errData.message || 'Login failed');
      }

      const data: LoginResponse = await response.json();
      const userData: User = {
        user_id: data.user_id,
        name: data.name,
        role: data.role,
      };
      setUser(userData);
      localStorage.setItem('authToken', data.access_token);
      localStorage.setItem('userData', JSON.stringify(userData)); // Store user data in localStorage
      return true;
    } catch (err) {
      console.log(err, 'Login failed');
      return false;
    }
  };

  const logout = () => {
    setUser(null);
    localStorage.removeItem('authToken');
    localStorage.removeItem('userData'); // Remove user data from localStorage
  };

  return (
    <AuthContext.Provider value={{ user, login, logout }}>
      {children}
    </AuthContext.Provider>
  );
};

export const useAuth = (): AuthContextType => {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
};